/*
 * $Id$
 *
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 *
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.sarl.lang.compiler.batch;

import com.google.common.base.Strings;
import org.eclipse.xtext.util.Files;

/** The policy that should be used for cleaning the files after a compilation.
 *
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version batchcompiler 0.15.1 20250911-224823
 * @mavengroupid io.sarl.lang
 * @mavenartifactid batchcompiler
 * @since 0.10
 */
public enum CleaningPolicy {

	/**
	 * No cleaning. All the files that are generated by the SARL compilation stay as they are.
	 */
	NO_CLEANING,

	/**
	 * The subfolders of the root temporary folder will be cleaning with the
	 * {@link Files#cleanFolder(java.io.File, java.io.FileFilter, boolean, boolean)}
	 * function, with no file filtering.
	 */
	INTERNAL_CLEANING,

	/**
	 * All the files and the root temp directory are removed.
	 */
	FULL_CLEANING;

	/** Replies the default cleaning policy.
	 *
	 * @return the default cleaning policy.
	 */
	public static CleaningPolicy getDefault() {
		return FULL_CLEANING;
	}

	/** Parse the given case insensitive string for obtaining the cleaning policy.
	 *
	 * @param name the string to parse.
	 * @return the cleaning policy, or {@code null} if the string cannot be parsed.
	 */
	public static CleaningPolicy valueOfCaseInsensitive(String name) {
		if (Strings.isNullOrEmpty(name)) {
			return null;
		}
		try {
			return valueOf(name.toUpperCase());
		} catch (Exception exception) {
			return null;
		}
	}

	/** Replies the string representation of this cleaning policy.
	 *
	 * @return the string representation.
	 */
	public String getCaseInsensitiveName() {
		return name().toLowerCase();
	}

}
