/*
 * $Id$
 *
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 *
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.sarl.lang.validation;

import java.text.MessageFormat;

import com.google.inject.Inject;
import org.eclipse.xtext.nodemodel.SyntaxErrorMessage;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.conversion.XbaseValueConverterService;
import org.eclipse.xtext.xtext.parser.CardinalityAwareSyntaxErrorMessageProvider;

import io.sarl.lang.services.SARLGrammarKeywordAccess;

/** Provider of messages for syntax errors.
 *
 * <p>This provider enhances the error messages when a keyword is misplaced.
 *
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version compiler 0.15.1 20250911-224823
 * @mavengroupid io.sarl.lang
 * @mavenartifactid compiler
 */
public class SARLSyntaxErrorMessageProvider extends CardinalityAwareSyntaxErrorMessageProvider {

	@Inject
	private SARLGrammarKeywordAccess grammarAccess;

	@Inject
	private XbaseValueConverterService converter;

	@Override
	public SyntaxErrorMessage getSyntaxErrorMessage(IParserErrorContext context) {
		if (context != null) {
			final var recognitionException = context.getRecognitionException();
			if (recognitionException != null && recognitionException.token != null) {
				final var text = recognitionException.token.getText();
				if (!Strings.isEmpty(text)) {
					if (this.grammarAccess.isPureKeyword(text)) {
						final var protectedText = this.converter.getQualifiedNameValueConverter().toString(text);
			            return new SyntaxErrorMessage(
			            		MessageFormat.format(Messages.SARLSyntaxErrorMessageProvider_0, text, protectedText),
			            		SyntaxIssueCodes.USED_RESERVED_KEYWORD,
			            		new String[] {text, protectedText});
					} else if (this.grammarAccess.isKeyword(text)) {
			            return new SyntaxErrorMessage(
			            		MessageFormat.format(Messages.SARLSyntaxErrorMessageProvider_1, text),
			            		SyntaxIssueCodes.USED_RESERVED_KEYWORD);
					}
				}
			}
		}
		return super.getSyntaxErrorMessage(context);
	}

}
