/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.api.experienceindex;

import io.sarl.lang.core.annotation.DefaultValue;
import io.sarl.lang.core.annotation.DefaultValueSource;
import io.sarl.lang.core.annotation.DefaultValueUse;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSourceCode;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.lang.reflect.Array;
import java.net.InetSocketAddress;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.Random;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.crypto.Cipher;
import javax.crypto.spec.DESKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * This class computes the SARL Experience Index (SEI) of your machine.
 * 
 * <p>The SEI is inspirated by the <a href="https://en.wikipedia.org/wiki/Windows_System_Assessment_Tool">Windows
 * Experience Index</a>.
 * The SEI measures the capability of your computer's hardware and
 * software configuration and expresses this measurement as a number
 * called a base score. A higher base score generally means that your
 * computer will perform better and faster than a computer with a
 * lower base score.
 * 
 * <h2>Global Base Score</h2>
 * 
 * <p>The base score is a value between 1.0 and 5.9.
 * The base score is based on the sub scores of sub components.
 * 
 * <p>The model logic is tolerant of one subscore being below the threshold for
 * a particular level by 0.1.  For example, assume that in the above example,
 * the memory score was 4.0 and the processor score 3.9.
 * This would mean that the processor score would marginally be the only item
 * keeping the base score below level 4. The model addresses this issue by
 * rounding up a single value that is below the next round level by 0.1.
 * 
 * <p>After the sub scores are normalized, the global base score is the average
 * score of the sub components.
 * 
 * <h2>CPU Score</h2>
 * 
 * <p>The CPU score was created to measure the processor performance when tasked with common Janus
 * usage activities. The processor is assessed on the following items:<ol>
 * <li>Compression and decompression using the Zip compression algorithm</li>
 * <li>Encryption and decryption assessment</li>
 * <li>Arithmetic Operations</li>
 * </ol>
 * The results are normalized and weight averaged in order to arrive at the final CPU sub score.
 * 
 * <h2>Memory Score</h2>
 * 
 * <p>The memory score measures the bandwidth of moving data into and out of memory in
 * Mega Bytes per Second. The higher the bandwidth, the better the memory.
 * Not having enough memory is a limiting factor on performance. As a result,
 * the amount of memory in the system constrains the score value.
 * The amount of memory limits are:
 * <table>
 * <caption>Memory Scores</caption>
 * <thead>
 * <tr><th>Amount of memory</th><th>Highest possible score</th></tr>
 * </thead>
 * <tbody>
 * <tr><td>64 MB or less</td><td>1.0</td></tr>
 * <tr><td>Less than 128 MB</td><td>2.0</td></tr>
 * <tr><td>Less than 256 MB</td><td>3.0</td></tr>
 * <tr><td>Less than 512 MB</td><td>4.0</td></tr>
 * <tr><td>Less than 1 GB</td><td>5.0</td></tr>
 * </tbody>
 * </table>
 * 
 * <h2>Disk Score</h2>
 * 
 * <p>The disk score measures disk bandwidth (in Mega Bytes per Second). The conversion
 * to an index number is set up in a way that all modern disks will score at least 2.0.
 * 
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version api.experienceindex 0.15.1 20250911-224825
 * @mavengroupid io.sarl.sdk
 * @mavenartifactid api.experienceindex
 * @since 0.6
 */
@SarlSpecification("0.15")
@SarlElementType(10)
@XbaseGenerated
@SuppressWarnings("all")
public final class SarlExperienceIndex {
  /**
   * This class contains the different SEI of your machine.
   * 
   * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
   * @version api.experienceindex 0.15.1 20250911-224825
   * @mavengroupid io.sarl.sdk
   * @mavenartifactid api.experienceindex
   */
  @SarlSpecification("0.15")
  @SarlElementType(10)
  @XbaseGenerated
  public static class SEI {
    private float base = Float.NaN;

    private float cpu = Float.NaN;

    private float memory = Float.NaN;

    private float disk = Float.NaN;

    SEI() {
    }

    /**
     * Set the base score.
     * 
     * @param score
     */
    void setBaseScore(final float score) {
      this.base = score;
    }

    /**
     * Replies the base score.
     * 
     * @return the base score, or {@link Float#NaN} if
     * the score was not computed.
     */
    @Pure
    public float getBaseScore() {
      return this.base;
    }

    /**
     * Set the CPU score.
     * 
     * @param score
     */
    void setCpuScore(final float score) {
      this.cpu = score;
    }

    /**
     * Replies the CPU score.
     * 
     * @return the CPU score, or {@link Float#NaN} if
     * the score was not computed.
     */
    @Pure
    public float getCpuScore() {
      return this.cpu;
    }

    /**
     * Set the Memory score.
     * 
     * @param score
     */
    void setMemoryScore(final float score) {
      this.memory = score;
    }

    /**
     * Replies the Memory score.
     * 
     * @return the memory score, or {@link Float#NaN} if
     * the score was not computed.
     */
    @Pure
    public float getMemoryScore() {
      return this.memory;
    }

    /**
     * Set the disk score.
     * 
     * @param score
     */
    void setDiskScore(final float score) {
      this.disk = score;
    }

    /**
     * Replies the disk score.
     * 
     * @return the disk score, or {@link Float#NaN} if
     * the score was not computed.
     */
    @Pure
    public float getDiskScore() {
      return this.disk;
    }

    @Override
    @Pure
    @SyntheticMember
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      SEI other = (SEI) obj;
      if (Float.floatToIntBits(other.base) != Float.floatToIntBits(this.base))
        return false;
      if (Float.floatToIntBits(other.cpu) != Float.floatToIntBits(this.cpu))
        return false;
      if (Float.floatToIntBits(other.memory) != Float.floatToIntBits(this.memory))
        return false;
      if (Float.floatToIntBits(other.disk) != Float.floatToIntBits(this.disk))
        return false;
      return super.equals(obj);
    }

    @Override
    @Pure
    @SyntheticMember
    public int hashCode() {
      int result = super.hashCode();
      final int prime = 31;
      result = prime * result + Float.hashCode(this.base);
      result = prime * result + Float.hashCode(this.cpu);
      result = prime * result + Float.hashCode(this.memory);
      result = prime * result + Float.hashCode(this.disk);
      return result;
    }
  }

  /**
   * Version of the SEI.
   */
  public static final String VERSION = "1.0";

  private static final float LOWER_COMPRESSION_DELAY = SarlExperienceIndex.secs(3);

  private static final float LOWER_ENCRYPTION_DELAY = SarlExperienceIndex.secs(4);

  private static final float LOWER_ARITHMETIC_DELAY = SarlExperienceIndex.secs(30);

  private static final float LOWER_MEMORY_DELAY = SarlExperienceIndex.secs(1);

  private static final float LOWER_DISK_DELAY = SarlExperienceIndex.secs(5);

  private static final int HIGHER_PROCESSOR_COUNT = 16;

  private static final float NS = ((1000f * 1000f) * 1000f);

  @Pure
  private static float secs(final float s) {
    return (s * SarlExperienceIndex.NS);
  }

  @Pure
  private static float megaBytes(final float m) {
    return ((m * 1024) * 1024);
  }

  @Pure
  private static String wrap(final float v) {
    String _xifexpression = null;
    boolean _isNaN = Float.isNaN(v);
    if (_isNaN) {
      _xifexpression = Messages.SarlExperienceIndex_0;
    } else {
      _xifexpression = Float.toString(v);
    }
    return _xifexpression;
  }

  @Pure
  private static float clamp(final float v, final float min, final float max) {
    if ((v < min)) {
      return min;
    }
    if ((v > max)) {
      return max;
    }
    return v;
  }

  @Pure
  private static float normalize(final float value) {
    float _xblockexpression = (float) 0;
    {
      boolean _isNaN = Float.isNaN(value);
      if (_isNaN) {
        return Float.NaN;
      }
      int _round = Math.round((value * 10f));
      _xblockexpression = (_round / 10f);
    }
    return _xblockexpression;
  }

  private static void normalize(final float... scores) {
    for (int i = 0; (i < scores.length); i++) {
      {
        float s = scores[i];
        boolean _isNaN = Float.isNaN(s);
        if ((!_isNaN)) {
          for (int j = 0; (j < scores.length); j++) {
            if ((j != i)) {
              float score = scores[j];
              if (((s >= (score - 0.1f)) && (s <= score))) {
                s = score;
              }
            }
          }
        }
        scores[i] = s;
      }
    }
  }

  private static float avg(final float... scores) {
    int n = 0;
    float score = 0f;
    for (final float subscore : scores) {
      boolean _isNaN = Float.isNaN(subscore);
      if ((!_isNaN)) {
        score = (score + subscore);
        n++;
      }
    }
    if ((n > 0)) {
      score = (score / n);
      score = SarlExperienceIndex.normalize(score);
    } else {
      score = Float.NaN;
    }
    return score;
  }

  private static void garbage() {
    final Runtime r = Runtime.getRuntime();
    for (int i = 0; (i < 6); i++) {
      r.gc();
    }
  }

  @Pure
  private static SeiCanceler wrap(final SeiCanceler canceller) {
    SeiCanceler _xifexpression = null;
    if ((canceller == null)) {
      final SeiCanceler _function = () -> {
        return false;
      };
      _xifexpression = _function;
    } else {
      _xifexpression = canceller;
    }
    return _xifexpression;
  }

  private static final SarlExperienceIndex.SEI SEI_SINGLETON = new SarlExperienceIndex.SEI();

  private SarlExperienceIndex() {
  }

  /**
   * Run the SEI from the command-line.
   * 
   * @param args no used.
   */
  @SuppressWarnings("discouraged_reference")
  public static void main(final String... args) {
    InputOutput.<String>println(MessageFormat.format(Messages.SarlExperienceIndex_1, SarlExperienceIndex.VERSION));
    final SarlExperienceIndex.SEI SEI = SarlExperienceIndex.getJanusExperienceIndex();
    InputOutput.<String>println(MessageFormat.format(Messages.SarlExperienceIndex_2, SarlExperienceIndex.wrap(SEI.getBaseScore())));
    InputOutput.<String>println(MessageFormat.format(Messages.SarlExperienceIndex_3, Float.valueOf(SEI.getCpuScore())));
    InputOutput.<String>println(MessageFormat.format(Messages.SarlExperienceIndex_4, Float.valueOf(SEI.getMemoryScore())));
    InputOutput.<String>println(MessageFormat.format(Messages.SarlExperienceIndex_5, Float.valueOf(SEI.getDiskScore())));
    InputOutput.<String>println(Messages.SarlExperienceIndex_6);
  }

  /**
   * Replies the current SEI.
   * 
   * @param canceller the object that permits to stop the computation before its end.
   * @return the current SEI.
   */
  @DefaultValueSource
  @Pure
  public static SarlExperienceIndex.SEI getJanusExperienceIndex(@DefaultValue("io.sarl.api.experienceindex.SarlExperienceIndex#GETJANUSEXPERIENCEINDEX_0") final SeiCanceler canceller) {
    SarlExperienceIndex.SEI _xblockexpression = null;
    {
      SarlExperienceIndex.baseScore(canceller);
      _xblockexpression = SarlExperienceIndex.SEI_SINGLETON;
    }
    return _xblockexpression;
  }

  /**
   * Compute the global SEI.
   * 
   * @param canceller the object that permits to stop the computation before its end.
   * @return the global SEI.
   */
  @DefaultValueSource
  public static float baseScore(@DefaultValue("io.sarl.api.experienceindex.SarlExperienceIndex#BASESCORE_0") final SeiCanceler canceller) {
    SeiCanceler clr = SarlExperienceIndex.wrap(canceller);
    float score = SarlExperienceIndex.SEI_SINGLETON.getBaseScore();
    if (((!clr.isCanceled()) && Float.isNaN(score))) {
      float _cpuScore = SarlExperienceIndex.cpuScore(canceller);
      float _memoryScore = SarlExperienceIndex.memoryScore(canceller);
      float _diskScore = SarlExperienceIndex.diskScore(canceller);
      final List<Float> scores = Collections.<Float>unmodifiableList(CollectionLiterals.<Float>newArrayList(Float.valueOf(_cpuScore), Float.valueOf(_memoryScore), Float.valueOf(_diskScore)));
      boolean _isCanceled = clr.isCanceled();
      if (_isCanceled) {
        return Float.NaN;
      }
      SarlExperienceIndex.normalize(((float[])Conversions.unwrapArray(scores, float.class)));
      score = SarlExperienceIndex.avg(((float[])Conversions.unwrapArray(scores, float.class)));
      SarlExperienceIndex.SEI_SINGLETON.setCpuScore(((scores.get(0)) == null ? 0 : (scores.get(0)).floatValue()));
      SarlExperienceIndex.SEI_SINGLETON.setMemoryScore(((scores.get(1)) == null ? 0 : (scores.get(1)).floatValue()));
      SarlExperienceIndex.SEI_SINGLETON.setDiskScore(((scores.get(2)) == null ? 0 : (scores.get(2)).floatValue()));
      SarlExperienceIndex.SEI_SINGLETON.setBaseScore(score);
    }
    return score;
  }

  /**
   * Compute the CPU score.
   * 
   * @param canceller the object that permits to stop the computation before its end.
   * @return the CPU score.
   */
  @DefaultValueSource
  public static float cpuScore(@DefaultValue("io.sarl.api.experienceindex.SarlExperienceIndex#CPUSCORE_0") final SeiCanceler canceller) {
    final SeiCanceler clr = SarlExperienceIndex.wrap(canceller);
    float score = SarlExperienceIndex.SEI_SINGLETON.getCpuScore();
    if (((!clr.isCanceled()) && Float.isNaN(score))) {
      float _computeCompressionScore = SarlExperienceIndex.computeCompressionScore(canceller);
      float _computeEncryptionScore = SarlExperienceIndex.computeEncryptionScore(canceller);
      float _computeArithmeticScore = SarlExperienceIndex.computeArithmeticScore(canceller);
      float _computeMultiProcessorScore = SarlExperienceIndex.computeMultiProcessorScore(canceller);
      List<Float> scores = Collections.<Float>unmodifiableList(CollectionLiterals.<Float>newArrayList(Float.valueOf(_computeCompressionScore), Float.valueOf(_computeEncryptionScore), Float.valueOf(_computeArithmeticScore), Float.valueOf(_computeMultiProcessorScore)));
      final List<Float> _converted_scores = (List<Float>)scores;
      score = SarlExperienceIndex.avg(((float[])Conversions.unwrapArray(_converted_scores, float.class)));
      score = SarlExperienceIndex.normalize(score);
      SarlExperienceIndex.SEI_SINGLETON.setCpuScore(score);
    }
    return score;
  }

  private static float computeMultiProcessorScore(final SeiCanceler canceller) {
    float _xtrycatchfinallyexpression = (float) 0;
    try {
      float _xblockexpression = (float) 0;
      {
        Runtime r = Runtime.getRuntime();
        int n = r.availableProcessors();
        _xblockexpression = SarlExperienceIndex.clamp((((n * 5f) / SarlExperienceIndex.HIGHER_PROCESSOR_COUNT) + 1f), 1, 6);
      }
      _xtrycatchfinallyexpression = _xblockexpression;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        _xtrycatchfinallyexpression = Float.NaN;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return _xtrycatchfinallyexpression;
  }

  @SuppressWarnings("unused_local_variable")
  private static float computeArithmeticScore(final SeiCanceler canceller) {
    float _xtrycatchfinallyexpression = (float) 0;
    try {
      float _xblockexpression = (float) 0;
      {
        long s = System.nanoTime();
        for (int i = 0; (i < 100000000); i++) {
          {
            double atanResult = Math.atan2(123, 456);
            boolean _isCanceled = canceller.isCanceled();
            if (_isCanceled) {
              return Float.NaN;
            }
          }
        }
        long e = System.nanoTime();
        long arithDelay = (e - s);
        SarlExperienceIndex.garbage();
        float score = ((arithDelay * 5f) / SarlExperienceIndex.LOWER_ARITHMETIC_DELAY);
        score = (6 - score);
        _xblockexpression = SarlExperienceIndex.clamp(score, 1, 6);
      }
      _xtrycatchfinallyexpression = _xblockexpression;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        _xtrycatchfinallyexpression = Float.NaN;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return _xtrycatchfinallyexpression;
  }

  private static float computeEncryptionScore(final SeiCanceler canceller) {
    float _xtrycatchfinallyexpression = (float) 0;
    try {
      float _xblockexpression = (float) 0;
      {
        Random rnd = new Random();
        StringBuilder t = new StringBuilder();
        for (int i = 0; (i < 6000); i++) {
          for (int j = 0; (j < 1024); j++) {
            {
              int _nextInt = rnd.nextInt();
              t.append(((char) _nextInt));
              boolean _isCanceled = canceller.isCanceled();
              if (_isCanceled) {
                return Float.NaN;
              }
            }
          }
        }
        t.trimToSize();
        byte[] buffer = t.toString().getBytes();
        SarlExperienceIndex.garbage();
        Properties p = System.getProperties();
        String _property = p.getProperty("user.name");
        String _plus = (_property + "@");
        String _hostName = new InetSocketAddress(0).getHostName();
        String _plus_1 = ((_plus + _hostName) + ":jdk_");
        String _property_1 = p.getProperty("java.version");
        String _plus_2 = ((_plus_1 + _property_1) + ":os_");
        String _property_2 = p.getProperty("os.name");
        String _plus_3 = ((_plus_2 + _property_2) + "-");
        String _property_3 = p.getProperty("os.version");
        String seed = (_plus_3 + _property_3);
        byte[] original = SarlExperienceIndex.md5(seed).getBytes("UTF8");
        Object _newInstance = Array.newInstance(byte.class, DESKeySpec.DES_KEY_LEN);
        byte[] bkey = ((byte[]) _newInstance);
        for (int i = 0; (i < DESKeySpec.DES_KEY_LEN); i++) {
          {
            int _length = original.length;
            bkey[i] = original[(i % _length)];
            boolean _isCanceled = canceller.isCanceled();
            if (_isCanceled) {
              return Float.NaN;
            }
          }
        }
        SecretKeySpec kkey = new SecretKeySpec(bkey, "DES");
        Cipher cipher = Cipher.getInstance("DES");
        long s = System.nanoTime();
        MessageDigest.getInstance("MD5").digest(buffer);
        boolean _isCanceled = canceller.isCanceled();
        if (_isCanceled) {
          return Float.NaN;
        }
        MessageDigest.getInstance("SHA").digest(buffer);
        boolean _isCanceled_1 = canceller.isCanceled();
        if (_isCanceled_1) {
          return Float.NaN;
        }
        cipher.init(Cipher.ENCRYPT_MODE, kkey);
        boolean _isCanceled_2 = canceller.isCanceled();
        if (_isCanceled_2) {
          return Float.NaN;
        }
        byte[] output = cipher.doFinal(buffer);
        boolean _isCanceled_3 = canceller.isCanceled();
        if (_isCanceled_3) {
          return Float.NaN;
        }
        cipher.init(Cipher.DECRYPT_MODE, kkey);
        boolean _isCanceled_4 = canceller.isCanceled();
        if (_isCanceled_4) {
          return Float.NaN;
        }
        cipher.doFinal(output);
        boolean _isCanceled_5 = canceller.isCanceled();
        if (_isCanceled_5) {
          return Float.NaN;
        }
        long e = System.nanoTime();
        long encryptionDelay = (e - s);
        SarlExperienceIndex.garbage();
        float score = ((encryptionDelay * 5f) / SarlExperienceIndex.LOWER_ENCRYPTION_DELAY);
        score = (6 - score);
        _xblockexpression = SarlExperienceIndex.clamp(score, 1, 6);
      }
      _xtrycatchfinallyexpression = _xblockexpression;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        _xtrycatchfinallyexpression = Float.NaN;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return _xtrycatchfinallyexpression;
  }

  private static String md5(final String str) {
    String _xblockexpression = null;
    {
      if ((str == null)) {
        return "";
      }
      byte[] uniqueKey = str.getBytes();
      byte[] hash = null;
      try {
        hash = MessageDigest.getInstance("MD5").digest(uniqueKey);
      } catch (final Throwable _t) {
        if (_t instanceof NoSuchAlgorithmException) {
          final NoSuchAlgorithmException e = (NoSuchAlgorithmException)_t;
          throw new Error(Messages.SarlExperienceIndex_7, e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      StringBuilder hashString = new StringBuilder();
      for (int i = 0; (i < hash.length); i++) {
        {
          String hex = Integer.toHexString(hash[i]);
          int _length = hex.length();
          if ((_length == 1)) {
            hashString.append("0");
            int _length_1 = hex.length();
            hashString.append(hex.charAt((_length_1 - 1)));
          } else {
            int _length_2 = hex.length();
            hashString.append(hex.substring((_length_2 - 2)));
          }
        }
      }
      _xblockexpression = hashString.toString();
    }
    return _xblockexpression;
  }

  private static float computeCompressionScore(final SeiCanceler canceller) {
    float _xtrycatchfinallyexpression = (float) 0;
    try {
      float _xblockexpression = (float) 0;
      {
        Random rnd = new Random();
        StringBuilder t = new StringBuilder();
        for (int i = 0; (i < 6000); i++) {
          for (int j = 0; (j < 1024); j++) {
            {
              int _nextInt = rnd.nextInt();
              t.append(((char) _nextInt));
              boolean _isCanceled = canceller.isCanceled();
              if (_isCanceled) {
                return Float.NaN;
              }
            }
          }
        }
        t.trimToSize();
        byte[] buffer = t.toString().getBytes();
        SarlExperienceIndex.garbage();
        long compressionDelay = 0;
        ByteArrayOutputStream _byteArrayOutputStream = new ByteArrayOutputStream();
        ZipOutputStream zos = new ZipOutputStream(_byteArrayOutputStream);
        try {
          long s = System.nanoTime();
          ZipEntry _zipEntry = new ZipEntry("test.bin");
          zos.putNextEntry(_zipEntry);
          zos.write(buffer, 0, buffer.length);
          long e = System.nanoTime();
          compressionDelay = (e - s);
        } finally {
          zos.close();
        }
        boolean _isCanceled = canceller.isCanceled();
        if (_isCanceled) {
          return Float.NaN;
        }
        SarlExperienceIndex.garbage();
        float score = ((compressionDelay * 5f) / SarlExperienceIndex.LOWER_COMPRESSION_DELAY);
        score = (6 - score);
        _xblockexpression = SarlExperienceIndex.clamp(score, 1, 6);
      }
      _xtrycatchfinallyexpression = _xblockexpression;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        _xtrycatchfinallyexpression = Float.NaN;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return _xtrycatchfinallyexpression;
  }

  /**
   * Compute the Memory score.
   * 
   * @param canceller the object that permits to stop the computation before its end.
   * @return the Memory score.
   */
  @DefaultValueSource
  @SuppressWarnings("unused_local_variable")
  public static float memoryScore(@DefaultValue("io.sarl.api.experienceindex.SarlExperienceIndex#MEMORYSCORE_0") final SeiCanceler canceller) {
    final SeiCanceler clr = SarlExperienceIndex.wrap(canceller);
    float score = SarlExperienceIndex.SEI_SINGLETON.getMemoryScore();
    if (((!clr.isCanceled()) && Float.isNaN(score))) {
      long s = System.nanoTime();
      float _megaBytes = SarlExperienceIndex.megaBytes(10);
      Object _newInstance = Array.newInstance(byte.class, ((int) _megaBytes));
      byte[] tab = ((byte[]) _newInstance);
      for (int i = 0; (i < tab.length); i++) {
        {
          boolean _isCanceled = clr.isCanceled();
          if (_isCanceled) {
            return Float.NaN;
          }
          tab[i] = ((byte) 123);
          byte t = 0;
          if ((i > 0)) {
            t = tab[(i - 1)];
          }
          int _length = tab.length;
          if ((i < (_length - 1))) {
            t = tab[(i + 1)];
          }
        }
      }
      for (int i = 0; (i < 10000); i++) {
        {
          new String("ABSD");
          boolean _isCanceled = clr.isCanceled();
          if (_isCanceled) {
            return Float.NaN;
          }
        }
      }
      long e = System.nanoTime();
      SarlExperienceIndex.garbage();
      score = (((e - s) * 5f) / SarlExperienceIndex.LOWER_MEMORY_DELAY);
      score = (6 - score);
      Runtime r = Runtime.getRuntime();
      long _maxMemory = r.maxMemory();
      float _megaBytes_1 = SarlExperienceIndex.megaBytes(64);
      if ((_maxMemory <= _megaBytes_1)) {
        score = 1f;
      } else {
        long _maxMemory_1 = r.maxMemory();
        float _megaBytes_2 = SarlExperienceIndex.megaBytes(128);
        if ((_maxMemory_1 <= _megaBytes_2)) {
          score = SarlExperienceIndex.clamp(score, 1, 2);
        } else {
          long _maxMemory_2 = r.maxMemory();
          float _megaBytes_3 = SarlExperienceIndex.megaBytes(256);
          if ((_maxMemory_2 <= _megaBytes_3)) {
            score = SarlExperienceIndex.clamp(score, 1, 3);
          } else {
            long _maxMemory_3 = r.maxMemory();
            float _megaBytes_4 = SarlExperienceIndex.megaBytes(512);
            if ((_maxMemory_3 <= _megaBytes_4)) {
              score = SarlExperienceIndex.clamp(score, 1, 4);
            } else {
              long _maxMemory_4 = r.maxMemory();
              float _megaBytes_5 = SarlExperienceIndex.megaBytes(1024);
              if ((_maxMemory_4 <= _megaBytes_5)) {
                score = SarlExperienceIndex.clamp(score, 1, 5);
              } else {
                score = SarlExperienceIndex.clamp(score, 1, 6);
              }
            }
          }
        }
      }
      score = SarlExperienceIndex.normalize(score);
      SarlExperienceIndex.SEI_SINGLETON.setMemoryScore(score);
    }
    return score;
  }

  /**
   * Compute the disk score.
   * 
   * @param canceller the object that permits to stop the computation before its end.
   * @return the disk score.
   */
  @DefaultValueSource
  public static float diskScore(@DefaultValue("io.sarl.api.experienceindex.SarlExperienceIndex#DISKSCORE_0") final SeiCanceler canceller) {
    final SeiCanceler clr = SarlExperienceIndex.wrap(canceller);
    float score = SarlExperienceIndex.SEI_SINGLETON.getDiskScore();
    if (((!clr.isCanceled()) && Float.isNaN(score))) {
      try {
        File tempFile = File.createTempFile("SEI", ".bin");
        tempFile.deleteOnExit();
        long s = System.nanoTime();
        FileWriter fw = new FileWriter(tempFile);
        try {
          for (int i = 0; (i < ((1024 * 1024) * 20)); i++) {
            {
              fw.write("A");
              boolean _isCanceled = clr.isCanceled();
              if (_isCanceled) {
                return Float.NaN;
              }
            }
          }
          fw.flush();
        } finally {
          fw.close();
        }
        FileReader fr = new FileReader(tempFile);
        try {
          while ((fr.read() != (-1))) {
            boolean _isCanceled = clr.isCanceled();
            if (_isCanceled) {
              return Float.NaN;
            }
          }
        } finally {
          fr.close();
        }
        long e = System.nanoTime();
        tempFile.delete();
        score = (((e - s) * 5f) / SarlExperienceIndex.LOWER_DISK_DELAY);
        score = (6 - score);
        score = SarlExperienceIndex.clamp(score, 1, 6);
        score = SarlExperienceIndex.normalize(score);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          score = Float.NaN;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      SarlExperienceIndex.SEI_SINGLETON.setDiskScore(score);
    }
    return score;
  }

  /**
   * Default value for the parameter canceller
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static SeiCanceler $DEFAULT_VALUE$GETJANUSEXPERIENCEINDEX_0() {
    return null;
  }

  /**
   * Default value for the parameter canceller
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static SeiCanceler $DEFAULT_VALUE$BASESCORE_0() {
    return null;
  }

  /**
   * Default value for the parameter canceller
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static SeiCanceler $DEFAULT_VALUE$CPUSCORE_0() {
    return null;
  }

  /**
   * Default value for the parameter canceller
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static SeiCanceler $DEFAULT_VALUE$MEMORYSCORE_0() {
    return null;
  }

  /**
   * Default value for the parameter canceller
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static SeiCanceler $DEFAULT_VALUE$DISKSCORE_0() {
    return null;
  }

  /**
   * Replies the current SEI.
   * 
   * @optionalparam canceller the object that permits to stop the computation before its end.
   * @return the current SEI.
   */
  @DefaultValueUse("io.sarl.api.experienceindex.SeiCanceler")
  @SyntheticMember
  @Pure
  public static SarlExperienceIndex.SEI getJanusExperienceIndex() {
    return getJanusExperienceIndex($DEFAULT_VALUE$GETJANUSEXPERIENCEINDEX_0());
  }

  /**
   * Compute the global SEI.
   * 
   * @optionalparam canceller the object that permits to stop the computation before its end.
   * @return the global SEI.
   */
  @DefaultValueUse("io.sarl.api.experienceindex.SeiCanceler")
  @SyntheticMember
  public static float baseScore() {
    return baseScore($DEFAULT_VALUE$BASESCORE_0());
  }

  /**
   * Compute the CPU score.
   * 
   * @optionalparam canceller the object that permits to stop the computation before its end.
   * @return the CPU score.
   */
  @DefaultValueUse("io.sarl.api.experienceindex.SeiCanceler")
  @SyntheticMember
  public static float cpuScore() {
    return cpuScore($DEFAULT_VALUE$CPUSCORE_0());
  }

  /**
   * Compute the Memory score.
   * 
   * @optionalparam canceller the object that permits to stop the computation before its end.
   * @return the Memory score.
   */
  @DefaultValueUse("io.sarl.api.experienceindex.SeiCanceler")
  @SyntheticMember
  @SuppressWarnings("unused_local_variable")
  public static float memoryScore() {
    return memoryScore($DEFAULT_VALUE$MEMORYSCORE_0());
  }

  /**
   * Compute the disk score.
   * 
   * @optionalparam canceller the object that permits to stop the computation before its end.
   * @return the disk score.
   */
  @DefaultValueUse("io.sarl.api.experienceindex.SeiCanceler")
  @SyntheticMember
  public static float diskScore() {
    return diskScore($DEFAULT_VALUE$DISKSCORE_0());
  }
}
