/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.api.game.tgob;

import io.sarl.lang.core.annotation.DefaultValue;
import io.sarl.lang.core.annotation.DefaultValueSource;
import io.sarl.lang.core.annotation.DefaultValueUse;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSourceCode;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Default implementation of an action that could be done by the agent in the game time-based
 * goal-oriented behavior.
 * 
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version api.gametgob 0.15.1 20250911-224825
 * @mavengroupid io.sarl.sdk
 * @mavenartifactid api.gametgob
 * @since 0.13
 */
@SarlSpecification("0.15")
@SarlElementType(10)
@XbaseGenerated
@SuppressWarnings("all")
public class BaseAction extends AbstractGob implements Action {
  /**
   * Duration of the action in time units.
   */
  @Accessors
  private int duration;

  private final TreeMap<Goal, Double> insistences = CollectionLiterals.<Goal, Double>newTreeMap(((Comparator<Goal>) (Goal a, Goal b) -> {
    if ((a == b)) {
      return 0;
    }
    if ((a == null)) {
      return (-1);
    }
    if ((b == null)) {
      return 1;
    }
    int cmp = a.getName().compareTo(b.getName());
    if ((cmp != 0)) {
      return cmp;
    }
    return Integer.compare(System.identityHashCode(a), System.identityHashCode(b));
  }));

  /**
   * Constructor.
   * 
   * @param name the name of the action.
   * @param duration the duration of the action.
   */
  @DefaultValueSource
  public BaseAction(@DefaultValue("io.sarl.api.game.tgob.BaseAction#NEW_0") final String name, @DefaultValue("io.sarl.api.game.tgob.BaseAction#NEW_1") final int duration) {
    this.setName(name);
    this.setDuration(duration);
  }

  @Override
  @Pure
  public double getInsistenceChangeFor(final Goal goal) {
    if ((goal != null)) {
      final Double value = this.insistences.get(goal);
      if ((value != null)) {
        return ((value) == null ? 0 : (value).doubleValue());
      }
    }
    return 0.0;
  }

  /**
   * Change the insistence for the given goal when this action is applied.
   * 
   * @param goal the goal for which the insistence is changed.
   * @param insistence the new positive value of the insistence.
   */
  public void setInsistenceChangeFor(final Goal goal, final double insistence) {
    if ((goal != null)) {
      if ((insistence != 0.0)) {
        this.insistences.put(goal, Double.valueOf(insistence));
      } else {
        this.insistences.remove(goal);
      }
    }
  }

  @Override
  @Pure
  public String toString() {
    final StringBuilder msg = new StringBuilder();
    msg.append("Action: ").append(this.getName()).append("\t");
    boolean first = true;
    Set<Map.Entry<Goal, Double>> _entrySet = this.insistences.entrySet();
    for (final Map.Entry<Goal, Double> entry : _entrySet) {
      {
        final Goal goal = entry.getKey();
        final Double insistence = entry.getValue();
        if ((((insistence) == null ? 0 : (insistence).doubleValue()) != 0.0)) {
          if (first) {
            first = false;
          } else {
            msg.append(", ");
          }
          msg.append(goal.getName()).append(": ").append(AbstractGob.formatNumber(((insistence) == null ? 0 : (insistence).doubleValue())));
        }
      }
    }
    msg.append("\t").append(AbstractGob.formatNumber(this.getDuration())).append(" unit(s)");
    return msg.toString();
  }

  /**
   * Default value for the parameter name
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static String $DEFAULT_VALUE$NEW_0() {
    return null;
  }

  /**
   * Default value for the parameter duration
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("0")
  private static int $DEFAULT_VALUE$NEW_1() {
    return 0;
  }

  /**
   * Constructor.
   * 
   * @optionalparam name the name of the action.
   * @optionalparam duration the duration of the action.
   */
  @DefaultValueUse("java.lang.String,int")
  @SyntheticMember
  public BaseAction() {
    this($DEFAULT_VALUE$NEW_0(), $DEFAULT_VALUE$NEW_1());
  }

  /**
   * Constructor.
   * 
   * @optionalparam name the name of the action.
   * @optionalparam duration the duration of the action.
   */
  @DefaultValueUse("java.lang.String,int")
  @SyntheticMember
  public BaseAction(final int duration) {
    this($DEFAULT_VALUE$NEW_0(), duration);
  }

  /**
   * Constructor.
   * 
   * @optionalparam name the name of the action.
   * @optionalparam duration the duration of the action.
   */
  @DefaultValueUse("java.lang.String,int")
  @SyntheticMember
  public BaseAction(final String name) {
    this(name, $DEFAULT_VALUE$NEW_1());
  }

  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    BaseAction other = (BaseAction) obj;
    if (other.duration != this.duration)
      return false;
    return super.equals(obj);
  }

  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Integer.hashCode(this.duration);
    return result;
  }

  @Pure
  @Override
  public int getDuration() {
    return this.duration;
  }

  public void setDuration(final int duration) {
    this.duration = duration;
  }
}
