/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.api.game.tgob;

import io.sarl.lang.core.Agent;
import io.sarl.lang.core.Skill;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import java.util.Iterator;
import java.util.List;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Skill that implements the standard action selection algorithm that is defined for
 * the game time-based goal-oriented behaviors.
 * 
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version api.gametgob 0.15.1 20250911-224825
 * @mavengroupid io.sarl.sdk
 * @mavenartifactid api.gametgob
 * @since 0.13
 */
@SarlSpecification("0.15")
@SarlElementType(22)
@XbaseGenerated
@SuppressWarnings("all")
public class StandardActionSelectionSkill extends Skill implements SelectAction {
  @Override
  public Action selectAction(final List<? extends Action> actions, final List<? extends Goal> goals) {
    final Iterator<? extends Action> iterator = actions.iterator();
    Action bestAction = iterator.next();
    double bestValue = this.calculateDiscontentment(bestAction, goals);
    while (iterator.hasNext()) {
      {
        final Action action = iterator.next();
        double value = this.calculateDiscontentment(action, goals);
        if ((value < bestValue)) {
          bestValue = value;
          bestAction = action;
        }
      }
    }
    return bestAction;
  }

  /**
   * Calculate the discontentment for the given action
   */
  @Pure
  protected double calculateDiscontentment(final Action action, final List<? extends Goal> goals) {
    double discontentment = 0.0;
    for (final Goal goal : goals) {
      {
        double _insistence = goal.getInsistence();
        double _insistenceChangeFor = action.getInsistenceChangeFor(goal);
        double newInsistence = (_insistence + _insistenceChangeFor);
        int _duration = action.getDuration();
        double _insistenceChangePerTimeUnit = goal.getInsistenceChangePerTimeUnit();
        newInsistence = (newInsistence + (_duration * _insistenceChangePerTimeUnit));
        double _discontentmentFor = goal.getDiscontentmentFor(newInsistence);
        discontentment = (discontentment + _discontentmentFor);
      }
    }
    return discontentment;
  }

  @SyntheticMember
  public StandardActionSelectionSkill() {
    super();
  }

  @SyntheticMember
  public StandardActionSelectionSkill(final Agent arg0) {
    super(arg0);
  }
}
