/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.api.workingmemory;

import io.sarl.api.naming.name.ScopedDataName;
import io.sarl.lang.core.Capacity;
import io.sarl.lang.core.annotation.DefaultValue;
import io.sarl.lang.core.annotation.DefaultValueSource;
import io.sarl.lang.core.annotation.DefaultValueUse;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSourceCode;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Standard skill to access to the agent working memory.
 * In this skill, the knowledges are stored in a data structure
 * that is based on a {@code TreeMap}.
 * 
 * <p>This class is thread-safe.
 * 
 * @author <a href="https://github.com/stefanotedeschi">Stefano Tedeschi</a>
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version api.workingmemory 0.15.1 20250911-224825
 * @mavengroupid io.sarl.sdk
 * @mavenartifactid api.workingmemory
 * @since 0.15
 */
@SarlSpecification("0.15")
@SarlElementType(22)
@XbaseGenerated
@SuppressWarnings("all")
public class DictionaryWorkingMemory extends AbstractWorkingMemory implements Capacity {
  private static final String NULL_VALUE = "null value";

  private final TreeMap<ScopedDataName, Object> localState;

  private final AtomicBoolean enableNullValues;

  /**
   * Constructor.
   * 
   * @param initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @param comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @param enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueSource
  public DictionaryWorkingMemory(@DefaultValue("io.sarl.api.workingmemory.DictionaryWorkingMemory#NEW_0") final Map<ScopedDataName, Object> initialBase, @DefaultValue("io.sarl.api.workingmemory.DictionaryWorkingMemory#NEW_1") final Comparator<ScopedDataName> comparator, @DefaultValue("io.sarl.api.workingmemory.DictionaryWorkingMemory#NEW_2") final boolean enableNullValues) {
    if ((comparator == null)) {
      TreeMap<ScopedDataName, Object> _treeMap = new TreeMap<ScopedDataName, Object>();
      this.localState = _treeMap;
    } else {
      TreeMap<ScopedDataName, Object> _treeMap_1 = new TreeMap<ScopedDataName, Object>(comparator);
      this.localState = _treeMap_1;
    }
    AtomicBoolean _atomicBoolean = new AtomicBoolean(enableNullValues);
    this.enableNullValues = _atomicBoolean;
    if (((initialBase != null) && (!initialBase.isEmpty()))) {
      Set<Map.Entry<ScopedDataName, Object>> _entrySet = initialBase.entrySet();
      for (final Map.Entry<ScopedDataName, Object> entry : _entrySet) {
        {
          final ScopedDataName knowledge = entry.getKey();
          final Object value = entry.getValue();
          if ((knowledge != null)) {
            if (((value != null) || enableNullValues)) {
              this.localState.put(knowledge, value);
            }
          }
        }
      }
    }
  }

  @Pure
  @Override
  public boolean isNullAllowedInMemory() {
    return this.enableNullValues.get();
  }

  @Override
  public void setNullAllowedInMemory(final boolean enable) {
    this.enableNullValues.set(enable);
  }

  @Pure
  @Override
  public <T extends Object> T getKnowledge(final ScopedDataName id, final Class<T> type) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (id != null);
      }
    }
    assert new $AssertEvaluator$().$$result : "null id is not allowed";
    synchronized (this.getWorkingMemoryLock()) {
      final Function<ScopedDataName, Object> _function = (ScopedDataName it) -> {
        try {
          throw new KnowledgeMissingException();
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      Object value = this.localState.computeIfAbsent(id, _function);
      if ((value == DictionaryWorkingMemory.NULL_VALUE)) {
        value = null;
      }
      if ((type == null)) {
        return ((T) value);
      } else {
        return type.cast(value);
      }
    }
  }

  @Override
  public Object setKnowledge(final ScopedDataName id, final Object value) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (id != null);
      }
    }
    assert new $AssertEvaluator$().$$result : "null id is not allowed";
    synchronized (this.getWorkingMemoryLock()) {
      if (((value == null) && (!this.enableNullValues.get()))) {
        return this.localState.remove(id);
      }
      Object _xifexpression = null;
      if ((value == null)) {
        _xifexpression = DictionaryWorkingMemory.NULL_VALUE;
      } else {
        _xifexpression = value;
      }
      return this.localState.put(id, _xifexpression);
    }
  }

  @Override
  public void setKnowledgeIfAbsent(final ScopedDataName id, final Object value) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (id != null);
      }
    }
    assert new $AssertEvaluator$().$$result : "null id is not allowed";
    synchronized (this.getWorkingMemoryLock()) {
      if ((value != null)) {
        this.localState.putIfAbsent(id, value);
      }
    }
  }

  @Override
  public Object setKnowledgeIfPresent(final ScopedDataName id, final Object value) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (id != null);
      }
    }
    assert new $AssertEvaluator$().$$result : "null id is not allowed";
    synchronized (this.getWorkingMemoryLock()) {
      if (((value == null) && (!this.enableNullValues.get()))) {
        return this.localState.remove(id);
      }
      final Object oldValue = this.localState.get(id);
      if ((oldValue != null)) {
        Object _xifexpression = null;
        if ((value == null)) {
          _xifexpression = DictionaryWorkingMemory.NULL_VALUE;
        } else {
          _xifexpression = value;
        }
        this.localState.put(id, _xifexpression);
        Object _xifexpression_1 = null;
        if ((oldValue == DictionaryWorkingMemory.NULL_VALUE)) {
          _xifexpression_1 = null;
        } else {
          _xifexpression_1 = oldValue;
        }
        return _xifexpression_1;
      }
      return null;
    }
  }

  @Pure
  @Override
  public Stream<ScopedDataName> getDefinedForName(final String name) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(name);
        this.$$result = (!_isNullOrEmpty);
      }
    }
    assert new $AssertEvaluator$().$$result : "null name is not allowed";
    synchronized (this.getWorkingMemoryLock()) {
      final Predicate<ScopedDataName> _function = (ScopedDataName it) -> {
        String _name = it.getName();
        return Objects.equals(_name, name);
      };
      return this.localState.keySet().stream().filter(_function);
    }
  }

  @Pure
  @Override
  public boolean isDefined(final ScopedDataName id) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (id != null);
      }
    }
    assert new $AssertEvaluator$().$$result : "null id is not allowed";
    synchronized (this.getWorkingMemoryLock()) {
      return this.localState.containsKey(id);
    }
  }

  @Override
  public Object removeKnowledge(final ScopedDataName id) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (id != null);
      }
    }
    assert new $AssertEvaluator$().$$result : "null id is not allowed";
    synchronized (this.getWorkingMemoryLock()) {
      return this.localState.remove(id);
    }
  }

  @Override
  public void removeNullValuedKnowledges() {
    synchronized (this.getWorkingMemoryLock()) {
      final Iterator<Map.Entry<ScopedDataName, Object>> iterator = this.localState.entrySet().iterator();
      while (iterator.hasNext()) {
        {
          final Map.Entry<ScopedDataName, Object> entry = iterator.next();
          Object _value = entry.getValue();
          if ((_value == DictionaryWorkingMemory.NULL_VALUE)) {
            iterator.remove();
          }
        }
      }
    }
  }

  @Pure
  @Override
  public Stream<ScopedDataName> getDefinedNames() {
    synchronized (this.getWorkingMemoryLock()) {
      return this.localState.keySet().stream();
    }
  }

  @Pure
  @Override
  public long getMemorySize() {
    synchronized (this.getWorkingMemoryLock()) {
      return this.localState.size();
    }
  }

  @Pure
  @Override
  public boolean isMemoryEmpty() {
    synchronized (this.getWorkingMemoryLock()) {
      return this.localState.isEmpty();
    }
  }

  @Override
  public void clearMemory() {
    synchronized (this.getWorkingMemoryLock()) {
      this.localState.clear();
    }
  }

  /**
   * Default value for the parameter initialBase
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static Map $DEFAULT_VALUE$NEW_0() {
    return null;
  }

  /**
   * Default value for the parameter comparator
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static Comparator $DEFAULT_VALUE$NEW_1() {
    return null;
  }

  /**
   * Default value for the parameter enableNullValues
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("false")
  private static boolean $DEFAULT_VALUE$NEW_2() {
    return false;
  }

  /**
   * Constructor.
   * 
   * @optionalparam initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @optionalparam comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @optionalparam enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueUse("java.util.Map,java.util.Comparator,boolean")
  @SyntheticMember
  public DictionaryWorkingMemory() {
    this($DEFAULT_VALUE$NEW_0(), $DEFAULT_VALUE$NEW_1(), $DEFAULT_VALUE$NEW_2());
  }

  /**
   * Constructor.
   * 
   * @optionalparam initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @optionalparam comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @optionalparam enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueUse("java.util.Map,java.util.Comparator,boolean")
  @SyntheticMember
  public DictionaryWorkingMemory(final boolean enableNullValues) {
    this($DEFAULT_VALUE$NEW_0(), $DEFAULT_VALUE$NEW_1(), enableNullValues);
  }

  /**
   * Constructor.
   * 
   * @optionalparam initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @optionalparam comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @optionalparam enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueUse("java.util.Map,java.util.Comparator,boolean")
  @SyntheticMember
  public DictionaryWorkingMemory(final Comparator<ScopedDataName> comparator) {
    this($DEFAULT_VALUE$NEW_0(), comparator, $DEFAULT_VALUE$NEW_2());
  }

  /**
   * Constructor.
   * 
   * @optionalparam initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @optionalparam comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @optionalparam enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueUse("java.util.Map,java.util.Comparator,boolean")
  @SyntheticMember
  public DictionaryWorkingMemory(final Map<ScopedDataName, Object> initialBase) {
    this(initialBase, $DEFAULT_VALUE$NEW_1(), $DEFAULT_VALUE$NEW_2());
  }

  /**
   * Constructor.
   * 
   * @optionalparam initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @optionalparam comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @optionalparam enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueUse("java.util.Map,java.util.Comparator,boolean")
  @SyntheticMember
  public DictionaryWorkingMemory(final Comparator<ScopedDataName> comparator, final boolean enableNullValues) {
    this($DEFAULT_VALUE$NEW_0(), comparator, enableNullValues);
  }

  /**
   * Constructor.
   * 
   * @optionalparam initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @optionalparam comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @optionalparam enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueUse("java.util.Map,java.util.Comparator,boolean")
  @SyntheticMember
  public DictionaryWorkingMemory(final Map<ScopedDataName, Object> initialBase, final boolean enableNullValues) {
    this(initialBase, $DEFAULT_VALUE$NEW_1(), enableNullValues);
  }

  /**
   * Constructor.
   * 
   * @optionalparam initialBase the initial content of the working memory. If this argument is not provided,
   *     the memory is empty.
   * @optionalparam comparator the comparator for ensuring unique knowledge identifiers in the memory. If this argument
   *     is not provided, the natural order of {@code ScopedDataName} is used.
   * @optionalparam enableNullValues indicates if the {@code null} values are allowed to be assocated to knowledge in
   *     the working memory. Default value is {@code false}.
   */
  @DefaultValueUse("java.util.Map,java.util.Comparator,boolean")
  @SyntheticMember
  public DictionaryWorkingMemory(final Map<ScopedDataName, Object> initialBase, final Comparator<ScopedDataName> comparator) {
    this(initialBase, comparator, $DEFAULT_VALUE$NEW_2());
  }

  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    return super.equals(obj);
  }

  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    return result;
  }
}
