/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.api.workingmemory;

import io.sarl.api.naming.name.ScopedDataName;
import io.sarl.lang.core.AgentTrait;
import io.sarl.lang.core.Capacity;
import io.sarl.lang.core.DefaultSkill;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSpecification;
import java.util.List;
import java.util.stream.Stream;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Capacity to access to the agent working memory.
 * This working memory enables to have access to a part
 * of the data storage that is exclusively dedicated to
 * the agent.
 * 
 * <p>If the agent does not provide a specific skill for this capacity,
 * the skill to be used is {@link DictionaryWorkingMemory}.
 * 
 * @author <a href="https://github.com/stefanotedeschi">Stefano Tedeschi</a>
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version api.workingmemory 0.15.1 20250911-224825
 * @mavengroupid io.sarl.sdk
 * @mavenartifactid api.workingmemory
 * @since 0.15
 */
@DefaultSkill(DictionaryWorkingMemory.class)
@SarlSpecification("0.15")
@SarlElementType(20)
@XbaseGenerated
@SuppressWarnings("all")
public interface WorkingMemory extends Capacity {
  /**
   * Replies if the {@code null} value is allowed to be associated to a stored knowledge
   * in the working memory.
   * 
   * <p>If {@code null} value is not allowed in the memory, then each time a {@code null}
   * value is put in the memory for a knowledge, this knowledge is removed from the memory.
   * In the case the {@code null} value is allowed in the memory, then it is associated
   * to the knowledge.
   * 
   * <p>CAUTION: the support of the {@code null} value also depends on the concrete
   * implementation of the working memory. Please refer to the documentation of
   * the working memory's implementation for further details.
   * 
   * @return {@code true} if {@code null} value is allowed in working memory.
   */
  @Pure
  boolean isNullAllowedInMemory();

  /**
   * Change the flag indicating if the {@code null} value is allowed to be associated
   * to a stored knowledge in the working memory.
   * 
   * <p>If {@code null} value is not allowed in the memory, then each time a {@code null}
   * value is put in the memory for a knowledge, this knowledge is removed from the memory.
   * In the case the {@code null} value is allowed in the memory, then it is associated
   * to the knowledge.
   * 
   * <p>When this function is invoked with {@code false} as argument, the content of the
   * working memory is not changed. Therefore, the knowledges that
   * are already stored in the agent memory and associated with a {@code null} value
   * are kept in the memory storage. See {@link #removeNullValuedKnowledges()} for
   * removing {@code null} valued knowledges from the working memory.
   * 
   * <p>CAUTION: the support of the {@code null} value also depends on the concrete
   * implementation of the working memory. Please refer to the documentation of
   * the working memory's implementation for further details.
   * 
   * @param enable is {@code true} if {@code null} value is allowed in working memory;
   *     Otherwise, {@code null} value is not allowed.
   */
  void setNullAllowedInMemory(final boolean enable);

  /**
   * Replies the data value that is identified by the given id.
   * 
   * @param <T> the expected type of the data.
   * @param id the identifier of the knowledge to be extracted from the local state.
   * @param type the expected type of the data. If this type is not specified, the function
   *     replies the value as-is. Otherwise, if the value is not compatible with the
   *     given type, a {@code ClassCastException} is thrown.
   * @return the knowledge value.
   * @see #getKnowledge(ScopedDataName)
   * @throws KnowledgeMissingException when the knowledge is not stored in the working memory.
   */
  @Pure
  <T extends Object> T getKnowledge(final ScopedDataName id, final Class<T> type) throws KnowledgeMissingException;

  /**
   * Replies the knowledge value that is identified by the given id.
   * 
   * @param id the identifier of the knowledge to be extracted from the local state.
   * @return the knowledge value.
   * @see #getKnowledge(ScopedDataName, Class)
   * @throws KnowledgeMissingException when the knowledge is not stored in the working memory.
   */
  @Pure
  Object getKnowledge(final ScopedDataName id) throws KnowledgeMissingException;

  /**
   * Save the knowledge value in the working memory. If a value was
   * already stored in the working memory for this knowledge, then
   * the new value provided as argument will replace the previously
   * stored value.
   * 
   * @param id the identifier of the knowledge to be saved in the working memory.
   * @param value the new value of the knowledge.
   * @return previously stored knowledge value
   * @see #setKnowledgeIfAbsent(ScopedDataName, Object)
   * @see #setKnowledgeIfPresent(ScopedDataName, Object)
   */
  Object setKnowledge(final ScopedDataName id, final Object value);

  /**
   * Save the knowledge value in the working memory if this memory does not
   * already contain a value for this knowledge. If the memory
   * contain the given knowledge, this function has no effect.
   * 
   * @param id the identifier of the knowledge to be saved in the local state.
   * @param value the new value of the knowledge.
   * @see #setKnowledge(ScopedDataName, Object)
   * @see #setKnowledgeIfPresent(ScopedDataName, Object)
   */
  void setKnowledgeIfAbsent(final ScopedDataName id, final Object value);

  /**
   * Save the knowledge value in the working memory if this memory
   * already contains a value for this knowledge. If the memory does
   * not contain the given knowledge, this function has no effect.
   * 
   * @param id the identifier of the knowledge to be saved in the local state.
   * @param value the new value of the knowledge.
   * @return previously stored knowledge value
   * @see #setKnowledge(ScopedDataName, Object)
   * @see #setKnowledgeIfAbsent(ScopedDataName, Object)
   */
  Object setKnowledgeIfPresent(final ScopedDataName id, final Object value);

  /**
   * Replies all the scopes that are known in the working memory and associated
   * to the given knowledge name.
   * Since multiple knowledge may have the same name, the scopes of the knowledges
   * are used to make them unique in the working memory. This function replies
   * all the knowledges with the same name that are stored in the working memory
   * whatever their scopes.
   * 
   * <p>CAUTION: The replied stream must be manually synchronized by the caller of this function.
   * 
   * @param name the base name of the knowledge to search for.
   * @return the scopes of the knowledges with the given name in a stream.
   */
  @Pure
  Stream<ScopedDataName> getDefinedForName(final String name);

  /**
   * Replies all the scopes that are known in the working memory and associated
   * to the given knowledge name.
   * Since multiple knowledge may have the same name, the scopes of the knowledges
   * are used to make them unique in the working memory. This function replies
   * all the knowledges with the same name that are stored in the working memory
   * whatever their scopes.
   * 
   * @param name the base name of the knowledge to search for.
   * @return the scopes of the knowledges with the given name in a list.
   */
  @Pure
  List<ScopedDataName> getDefinedListForName(final String name);

  /**
   * Replies if the given knowledge corresponds to a knowledge that is
   * stored in the working memory.
   * 
   * <p>Definition of a knowledge is not related to its value. Therefore,
   * if the value of the stored knowledge is equal to {@code null}, the
   * knowledge is defined with a {@code null} value.
   * 
   * @param id the name to search for.
   * @return {@code true} if a knowledge is stored in the working memory. Otherwise
   *     {@code false}.
   */
  @Pure
  boolean isDefined(final ScopedDataName id);

  /**
   * Remove the knowledge from the working memory.
   * 
   * @param id the name to search for.
   * @return the value that was stored in the working memory.
   */
  Object removeKnowledge(final ScopedDataName id);

  /**
   * Remove from the working memory any knowledge that is associated to a {@code null}
   * value.
   */
  void removeNullValuedKnowledges();

  /**
   * Replies a resource that enables to have synchronized access to the
   * working memory. This resource could be used in a {@code synchronized}
   * block of code:
   * 
   * <pre><code>
   * synchronized(workingMemory.workingMemoryLock) {
   *    ...
   * }
   * </code></pre>
   * 
   * @return the synchronization resource.
   */
  @Pure
  Object getWorkingMemoryLock();

  /**
   * Replies all the defined names stored in the working memory.
   * 
   * <p>CAUTION: The replied stream must be manually synchronized by the caller of this function.
   * 
   * @return the stored knowledge names in a stream.
   */
  @Pure
  Stream<ScopedDataName> getDefinedNames();

  /**
   * Replies all the defined names stored in the working memory.
   * 
   * @return the stored knowledge names in an iterable object.
   */
  @Pure
  Iterable<ScopedDataName> getDefinedNameList();

  /**
   * Replies the number of knowledges that are stored in the working memory.
   * 
   * @return the number of knowledges.
   */
  @Pure
  long getMemorySize();

  /**
   * Replies if the working memory is empty or not.
   * The working memory is empty if there is no stored knowledge inside.
   * 
   * @return {@code true} if the working memory is empty.
   */
  @Pure
  boolean isMemoryEmpty();

  /**
   * Remove all knowledge from the working memory.
   */
  void clearMemory();

  /**
   * @ExcludeFromApidoc
   */
  class ContextAwareCapacityWrapper<C extends WorkingMemory> extends Capacity.ContextAwareCapacityWrapper<C> implements WorkingMemory {
    public ContextAwareCapacityWrapper(final C capacity, final AgentTrait caller) {
      super(capacity, caller);
    }

    public boolean isNullAllowedInMemory() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isNullAllowedInMemory();
      } finally {
        resetCallerInLocalThread();
      }
    }

    public void setNullAllowedInMemory(final boolean enable) {
      try {
        ensureCallerInLocalThread();
        this.capacity.setNullAllowedInMemory(enable);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public <T extends Object> T getKnowledge(final ScopedDataName id, final Class<T> type) throws KnowledgeMissingException {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getKnowledge(id, type);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Object getKnowledge(final ScopedDataName id) throws KnowledgeMissingException {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getKnowledge(id);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Object setKnowledge(final ScopedDataName id, final Object value) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.setKnowledge(id, value);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public void setKnowledgeIfAbsent(final ScopedDataName id, final Object value) {
      try {
        ensureCallerInLocalThread();
        this.capacity.setKnowledgeIfAbsent(id, value);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Object setKnowledgeIfPresent(final ScopedDataName id, final Object value) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.setKnowledgeIfPresent(id, value);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Stream<ScopedDataName> getDefinedForName(final String name) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefinedForName(name);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public List<ScopedDataName> getDefinedListForName(final String name) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefinedListForName(name);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public boolean isDefined(final ScopedDataName id) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isDefined(id);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Object removeKnowledge(final ScopedDataName id) {
      try {
        ensureCallerInLocalThread();
        return this.capacity.removeKnowledge(id);
      } finally {
        resetCallerInLocalThread();
      }
    }

    public void removeNullValuedKnowledges() {
      try {
        ensureCallerInLocalThread();
        this.capacity.removeNullValuedKnowledges();
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Object getWorkingMemoryLock() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getWorkingMemoryLock();
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Stream<ScopedDataName> getDefinedNames() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefinedNames();
      } finally {
        resetCallerInLocalThread();
      }
    }

    public Iterable<ScopedDataName> getDefinedNameList() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getDefinedNameList();
      } finally {
        resetCallerInLocalThread();
      }
    }

    public long getMemorySize() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.getMemorySize();
      } finally {
        resetCallerInLocalThread();
      }
    }

    public boolean isMemoryEmpty() {
      try {
        ensureCallerInLocalThread();
        return this.capacity.isMemoryEmpty();
      } finally {
        resetCallerInLocalThread();
      }
    }

    public void clearMemory() {
      try {
        ensureCallerInLocalThread();
        this.capacity.clearMemory();
      } finally {
        resetCallerInLocalThread();
      }
    }
  }
}
