/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.sre.janus.network.boot.configs;

import io.bootique.annotation.BQConfig;
import io.bootique.annotation.BQConfigProperty;
import io.bootique.config.ConfigurationFactory;
import io.sarl.lang.core.annotation.Injectable;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import io.sarl.sre.janus.boot.configs.SreConfig;
import jakarta.inject.Inject;
import java.util.ArrayList;
import java.util.Objects;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Configuration for the networking features of the SRE.
 * 
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version janus.network 3.0.15.1 20250911-224826
 * @mavengroupid io.sarl.sre.janus
 * @mavenartifactid janus.network
 * @since 0.12
 */
@BQConfig("Configuration for the SRE networking feature")
@SarlSpecification("0.15")
@SarlElementType(10)
@Injectable
@XbaseGenerated
@SuppressWarnings("all")
public class SreNetworkConfig {
  /**
   * Default name of the program.
   */
  public static final String DEFAULT_PROGRAM_NAME = "janusnode";

  /**
   * Prefix for the configuration entries of the modules.
   */
  public static final String PREFIX = "srenetwork";

  /**
   * Name of the property that enable or disable the networking features.
   */
  public static final String ENABLE_NAME = (SreNetworkConfig.PREFIX + ".enable");

  /**
   * Default value for the enabling state of the networking feature outside
   * the Eclipse environment.
   */
  public static final boolean DEFAULT_ENABLE_VALUE = false;

  /**
   * Name of the property that contains the name of the Hazelcast cluster.
   */
  public static final String CLUSTER_NAME_NAME = (SreNetworkConfig.PREFIX + ".clusterName");

  /**
   * Prefix for the default cluster name.
   */
  public static final String CLUSTER_NAME_PREFIX = "io.sarl.sre-hazelcast-";

  /**
   * Default value for the explicit cluster name.
   */
  public static final String DEFAULT_CLUSTER_NAME_VALUE = "";

  /**
   * Name of the property that indicates the method for cluster joins.
   */
  public static final String JOIN_METHOD_NAME = (SreNetworkConfig.PREFIX + ".joinMethod");

  /**
   * Name of the property that enable or disable the auto-increment of the port numbers.
   */
  public static final String PORT_AUTO_INCREMENT_NAME = (SreNetworkConfig.PREFIX + ".portAutoIncrement");

  /**
   * Default flag for the auto-increment of the port numbers.
   */
  public static final boolean DEFAULT_PORT_AUTO_INCREMENT_VALUE = true;

  /**
   * Name of the property that contains the minimal number of hazelcast nodes before starting.
   */
  public static final String MIN_CLUSTER_SIZE_NAME = (SreNetworkConfig.PREFIX + ".minClusterSize");

  /**
   * Default value for the minimal number of hazelcast nodes before starting.
   */
  public static final int DEFAULT_MIN_CLUSTER_SIZE_VALUE = 1;

  public static final String IP_LIST_CLUSTER = (SreNetworkConfig.PREFIX + ".clusterMemberIPs");

  /**
   * Default value for the eArray of Cluster members IP addresses represented as Strings.
   */
  public static final String DEFAULT_IP_LIST_CLUSTER = "127.0.0.1";

  /**
   * Name of the property that enable or disable the special configuration for
   * a local instance of Hazelcast. If this property is evaluated to {@code true},
   * it overrides all the other hazelcast configuration.
   */
  public static final String LOCAL_HAZELCAST_INSTANCE_NAME = (SreNetworkConfig.PREFIX + ".localHazelcastInstance");

  /**
   * Default value of the property that enable or disable the special configuration for
   * a local instance of Hazelcast. If this property is evaluated to {@code true},
   * it overrides all the other hazelcast configuration.
   */
  public static final boolean DEFAULT_LOCAL_HAZELCAST_INSTANCE_VALUE = false;

  /**
   * Regexp of a quarter of an IP adress
   */
  private static final String ZERO_TO_255 = "([01]?[0-9]{1,2}|2[0-4][0-9]|25[0-5])";

  /**
   * Regexp of a full IP address
   */
  private static final String IP_REGEXP = ((((((SreNetworkConfig.ZERO_TO_255 + "\\.") + SreNetworkConfig.ZERO_TO_255) + "\\.") + SreNetworkConfig.ZERO_TO_255) + "\\.") + SreNetworkConfig.ZERO_TO_255);

  /**
   * Regexp of IP list separator
   */
  private static final String IP_SEPARATOR_REGEXP = "[^0-9\\.]+";

  /**
   * Default separator for IP list separator
   */
  private static final String IP_SEPARATOR = ",";

  private boolean enable = SreNetworkConfig.DEFAULT_ENABLE_VALUE;

  private boolean localHazelcastInstance = SreNetworkConfig.DEFAULT_LOCAL_HAZELCAST_INSTANCE_VALUE;

  private String clusterName;

  private JoinMethod joinMethod;

  private String memberList;

  private String[] memberListArray;

  @Accessors(AccessorType.PUBLIC_GETTER)
  private boolean portAutoIncrement = SreNetworkConfig.DEFAULT_PORT_AUTO_INCREMENT_VALUE;

  @Accessors(AccessorType.PUBLIC_GETTER)
  private int minClusterSize = SreNetworkConfig.DEFAULT_MIN_CLUSTER_SIZE_VALUE;

  private SreConfig generalConfiguration;

  @Inject
  public SreNetworkConfig(final SreConfig generalConfiguration) {
    this.generalConfiguration = generalConfiguration;
  }

  /**
   * Replies the flag that enable or disable the networking features.
   * 
   * @return the enabling state of the enabling state of the networking extension.
   */
  @Pure
  public boolean getEnable() {
    return this.enable;
  }

  /**
   * Change the flag that enable or disable the networking features.
   * 
   * @param enable the flag value.
   */
  @BQConfigProperty("Enable or disable the networking features.")
  public void setEnable(final boolean enable) {
    this.enable = enable;
  }

  /**
   * Replies the property that enable or disable the special configuration for
   * a local instance of Hazelcast. If this property is evaluated to {@code true},
   * it overrides all the other hazelcast configuration.
   * 
   * @return the enabling state for local Hazelcast instance.
   */
  @Pure
  public boolean getLocalHazelcastInstance() {
    return this.localHazelcastInstance;
  }

  /**
   * Change the property that enable or disable the special configuration for
   * a local instance of Hazelcast. If this property is evaluated to {@code true},
   * it overrides all the other hazelcast configuration.
   * 
   * @param enable the enabling state.
   */
  @BQConfigProperty("Enable or disable the local Hazelcast instance.")
  public void setLocalHazelcastInstance(final boolean enable) {
    this.localHazelcastInstance = enable;
  }

  /**
   * Replies a cluster name following the standard naming convention.
   * 
   * @param rootContextId is the identifier of the root context of the SRE.
   * @return the cluster name.
   */
  @Pure
  public static String createStandardClusterName(final UUID rootContextId) {
    return SreNetworkConfig.createStandardClusterName(rootContextId.toString());
  }

  /**
   * Replies a cluster name following the standard naming convention.
   * 
   * @param rootContextId is the identifier of the root context of the SRE.
   * @return the cluster name.
   */
  @Pure
  public static String createStandardClusterName(final String rootContextId) {
    return (SreNetworkConfig.CLUSTER_NAME_PREFIX + rootContextId);
  }

  /**
   * Replies the name of the Hazelcast cluster name that is explicitly provided into the configuration.
   * 
   * @return the cluster name.
   */
  @Pure
  public String getClusterName() {
    String explicitName = this.clusterName;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(explicitName);
    if (_isNullOrEmpty) {
      final UUID id = this.generalConfiguration.getBoot().getRootContextID();
      explicitName = SreNetworkConfig.createStandardClusterName(id);
      this.clusterName = explicitName;
    }
    return explicitName;
  }

  /**
   * Replies the list of IP addresses of Hazelcast Cluster members as Strings.
   * 
   * @return the list of IP addresses of Hazelcast Cluster members.
   * @see #setClusterMemberIPs(String)
   * @see #setClusterMemberIPsArray(String[])
   * @see #getClusterMemberIPsArray()
   */
  @Pure
  public String getClusterMemberIPs() {
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(this.memberList);
    if (_isNullOrEmpty) {
      this.memberList = SreNetworkConfig.DEFAULT_IP_LIST_CLUSTER;
    }
    return this.memberList;
  }

  /**
   * Replies the list of IP addresses of Hazelcast Cluster members as Strings.
   * 
   * @return the list of IP addresses of Hazelcast Cluster members.
   * @see #getClusterMemberIPs()
   * @see #setClusterMemberIPs(String)
   * @see #setClusterMemberIPsArray(String[])
   */
  @Pure
  public String[] getClusterMemberIPsArray() {
    if ((this.memberListArray == null)) {
      final ArrayList<String> list = CollectionLiterals.<String>newArrayList();
      final Pattern pattern = Pattern.compile(SreNetworkConfig.IP_REGEXP);
      String[] _split = this.getClusterMemberIPs().trim().split(SreNetworkConfig.IP_SEPARATOR_REGEXP);
      for (final String ip : _split) {
        {
          final Matcher matcher = pattern.matcher(ip);
          boolean _matches = matcher.matches();
          if (_matches) {
            list.add(ip);
          }
        }
      }
      final String[] tab = new String[list.size()];
      list.<String>toArray(tab);
      this.memberListArray = tab;
    }
    return this.memberListArray;
  }

  /**
   * Set the list of IP addresses of Hazelcast Cluster members as Strings.
   * 
   * @param ips the list of IP addresses of Hazelcast Cluster members as Strings.
   * @see #getClusterMemberIPs()
   * @see #getClusterMemberIPsArray()
   * @see #setClusterMemberIPsArray(String[])
   */
  @BQConfigProperty("Specify the list of the IP adresses for the members that are allowed to be in the cluster. This configuration is used only if the join method is configured to TCP/IP.")
  public String[] setClusterMemberIPs(final String ips) {
    String[] _xblockexpression = null;
    {
      this.memberList = ips;
      _xblockexpression = this.memberListArray = null;
    }
    return _xblockexpression;
  }

  /**
   * Set the list of IP addresses of Hazelcast Cluster members as Strings.
   * 
   * @param ips the list of IP addresses of Hazelcast Cluster members as Strings.
   * @see #getClusterMemberIPs()
   * @see #setClusterMemberIPs(String)
   * @see #getClusterMemberIPsArray()
   */
  public String[] setClusterMemberIPsArray(final String... ips) {
    String[] _xblockexpression = null;
    {
      final StringBuilder buf = new StringBuilder();
      final Pattern pattern = Pattern.compile(SreNetworkConfig.IP_REGEXP);
      final ArrayList<String> list = CollectionLiterals.<String>newArrayList();
      for (final String ip : ips) {
        {
          final Matcher matcher = pattern.matcher(ip);
          boolean _matches = matcher.matches();
          if (_matches) {
            int _length = buf.length();
            if ((_length > 0)) {
              buf.append(SreNetworkConfig.IP_SEPARATOR);
            }
            buf.append(ip);
            list.add(ip);
          }
        }
      }
      this.memberList = buf.toString();
      final String[] tab = new String[list.size()];
      list.<String>toArray(tab);
      _xblockexpression = this.memberListArray = tab;
    }
    return _xblockexpression;
  }

  /**
   * Validate the given string of character for being a list of IP addresses.
   * 
   * @param ips is the list of IPs to check.
   * @return {@code true} if the given {@code ips} is valid.
   */
  public static boolean validateClusterMemberIPsArray(final String ips) {
    final String[] ipsArray = ips.trim().split(SreNetworkConfig.IP_SEPARATOR_REGEXP);
    final Pattern pattern = Pattern.compile(SreNetworkConfig.IP_REGEXP);
    for (final String ip : ipsArray) {
      {
        final Matcher matcher = pattern.matcher(ip);
        boolean _matches = matcher.matches();
        if ((!_matches)) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Change the name of the Hazelcast cluster name.
   * 
   * @param name the new cluster name.
   * @see #getClusterName()
   */
  @BQConfigProperty("Name of the Hazelcast cluster. By default, this name is based on the identifier of the agent root context.")
  public void setClusterName(final String name) {
    this.clusterName = name;
  }

  /**
   * Replies the method for cluster joining.
   * 
   * @return the join method.
   */
  @Pure
  public JoinMethod getJoinMethod() {
    this.ensureJoinMethod();
    return this.joinMethod;
  }

  /**
   * Set the method for cluster joining.
   * 
   * @param type the type of join method.
   */
  @BQConfigProperty("Specify the method for joining the Hazelcast cluster over the computer network")
  public void setJoinMethod(final JoinMethod type) {
    this.joinMethod = type;
    this.ensureJoinMethod();
  }

  /**
   * Set the minimum number of nodes into the cluster before starting the SRE.
   * 
   * @param clusterSize the minimum number of nodes into the cluster.
   */
  @BQConfigProperty("Specify the minimal number of nodes to wait for starting in the cluster before really launching the SARL Run-time Environment")
  public int setMinClusterSize(final int clusterSize) {
    int _xifexpression = (int) 0;
    if ((clusterSize < 1)) {
      _xifexpression = this.minClusterSize = 1;
    } else {
      _xifexpression = this.minClusterSize = clusterSize;
    }
    return _xifexpression;
  }

  private void ensureJoinMethod() {
    if ((this.joinMethod == null)) {
      this.joinMethod = JoinMethod.getDefault();
    }
  }

  /**
   * Change the flag that enable or disable the auto-increment of the port numbers.
   * 
   * @param enable the flag value.
   */
  @BQConfigProperty("Enable or disable the auto-incrementation of the Hazelcast port numbers.")
  public void setPortAutoIncrement(final boolean enable) {
    this.portAutoIncrement = enable;
  }

  /**
   * Replies the configuration for the networking features of the SRE.
   * 
   * @param configFactory the general configuration factory.
   * @return the configuration.
   */
  @Pure
  public static SreNetworkConfig getConfiguration(final ConfigurationFactory configFactory) {
    class $AssertEvaluator$ {
      final boolean $$result;
      $AssertEvaluator$() {
        this.$$result = (configFactory != null);
      }
    }
    assert new $AssertEvaluator$().$$result;
    SreNetworkConfig config = configFactory.<SreNetworkConfig>config(SreNetworkConfig.class, SreNetworkConfig.PREFIX);
    return config;
  }

  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    SreNetworkConfig other = (SreNetworkConfig) obj;
    if (other.enable != this.enable)
      return false;
    if (other.localHazelcastInstance != this.localHazelcastInstance)
      return false;
    if (!Objects.equals(this.clusterName, other.clusterName))
      return false;
    if (!Objects.equals(this.memberList, other.memberList))
      return false;
    if (other.portAutoIncrement != this.portAutoIncrement)
      return false;
    if (other.minClusterSize != this.minClusterSize)
      return false;
    return super.equals(obj);
  }

  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Boolean.hashCode(this.enable);
    result = prime * result + Boolean.hashCode(this.localHazelcastInstance);
    result = prime * result + Objects.hashCode(this.clusterName);
    result = prime * result + Objects.hashCode(this.memberList);
    result = prime * result + Boolean.hashCode(this.portAutoIncrement);
    result = prime * result + Integer.hashCode(this.minClusterSize);
    return result;
  }

  @Pure
  public boolean isPortAutoIncrement() {
    return this.portAutoIncrement;
  }

  @Pure
  public int getMinClusterSize() {
    return this.minClusterSize;
  }
}
