/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.sre.janus.network.boot.internal;

import com.hazelcast.config.Config;
import com.hazelcast.config.MulticastConfig;
import com.hazelcast.config.NetworkConfig;
import com.hazelcast.config.TcpIpConfig;
import com.hazelcast.core.Hazelcast;
import com.hazelcast.core.HazelcastInstance;
import io.bootique.BQModule;
import io.bootique.di.Binder;
import io.bootique.di.Injector;
import io.bootique.di.Provides;
import io.sarl.api.core.spaces.EventTransportService;
import io.sarl.api.core.spaces.LocalEventTransportService;
import io.sarl.apputils.bootiqueapp.config.LogConfig;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import io.sarl.sre.janus.network.boot.configs.JoinMethod;
import io.sarl.sre.janus.network.boot.configs.SreNetworkConfig;
import io.sarl.sre.janus.network.services.HazelcastContextService;
import io.sarl.sre.janus.network.services.HazelcastEventTransportService;
import io.sarl.sre.janus.network.services.HazelcastSpaceRepository;
import io.sarl.sre.janus.services.context.ContextService;
import io.sarl.sre.janus.services.context.LocalSpaceRepository;
import io.sarl.sre.janus.services.context.MemoryBasedContextService;
import io.sarl.sre.janus.services.context.SpaceRepository;
import io.sarl.sre.janus.services.logging.LoggingService;
import jakarta.inject.Singleton;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.function.Supplier;
import java.util.logging.Level;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * The module for injected the network features of Janus.
 * 
 * @author <a href="http://www.ciad-lab.fr/nicolas_gaud">Nicolas Gaud</a>
 * @version janus.network 3.0.15.1 20250911-224826
 * @mavengroupid io.sarl.sre.janus
 * @mavenartifactid janus.network
 * @since 0.12
 */
@SarlSpecification("0.15")
@SarlElementType(10)
@XbaseGenerated
@SuppressWarnings("all")
public class NetworkModule implements BQModule {
  @Override
  public void configure(final Binder binder) {
  }

  @Provides
  @Pure
  public SpaceRepository provideSpaceRepository(final Injector injector, final SreNetworkConfig netConfig) {
    Class<? extends SpaceRepository> instanceType = null;
    boolean _enable = netConfig.getEnable();
    if (_enable) {
      instanceType = HazelcastSpaceRepository.class;
    } else {
      instanceType = LocalSpaceRepository.class;
    }
    return injector.getInstance(instanceType);
  }

  @Provides
  @Singleton
  @Pure
  public ContextService provideContextService(final Injector injector, final SreNetworkConfig netConfig) {
    Class<? extends ContextService> instanceType = null;
    boolean _enable = netConfig.getEnable();
    if (_enable) {
      instanceType = HazelcastContextService.class;
    } else {
      instanceType = MemoryBasedContextService.class;
    }
    return injector.getInstance(instanceType);
  }

  @Provides
  @Singleton
  @Pure
  public EventTransportService provideEventTransportService(final Injector injector, final SreNetworkConfig netConfig) {
    Class<? extends EventTransportService> instanceType = null;
    boolean _enable = netConfig.getEnable();
    if (_enable) {
      instanceType = HazelcastEventTransportService.class;
    } else {
      instanceType = LocalEventTransportService.class;
    }
    return injector.getInstance(instanceType);
  }

  @Provides
  @Singleton
  public HazelcastInstance provideHazelcastInstance(final LoggingService logger, final SreNetworkConfig netConfig, final LogConfig logConfig) {
    final Config config = new Config();
    final String cname = netConfig.getClusterName();
    config.setClusterName(cname);
    final Supplier<String> _function = () -> {
      return MessageFormat.format(Messages.NetworkModule_3, cname);
    };
    logger.getKernelLogger().log(Level.FINE, _function);
    boolean _localHazelcastInstance = netConfig.getLocalHazelcastInstance();
    if (_localHazelcastInstance) {
      MulticastConfig _multicastConfig = config.getNetworkConfig().getJoin().getMulticastConfig();
      _multicastConfig.setEnabled(false);
      TcpIpConfig _tcpIpConfig = config.getNetworkConfig().getJoin().getTcpIpConfig();
      _tcpIpConfig.setEnabled(true);
      TcpIpConfig _tcpIpConfig_1 = config.getNetworkConfig().getJoin().getTcpIpConfig();
      _tcpIpConfig_1.setMembers(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("127.0.0.1")));
    } else {
      final JoinMethod jmethod = netConfig.getJoinMethod();
      final Supplier<String> _function_1 = () -> {
        return MessageFormat.format(Messages.NetworkModule_4, jmethod);
      };
      logger.getKernelLogger().log(Level.FINE, _function_1);
      if ((jmethod == JoinMethod.MULTICAST)) {
        MulticastConfig _multicastConfig_1 = config.getNetworkConfig().getJoin().getMulticastConfig();
        _multicastConfig_1.setEnabled(true);
        TcpIpConfig _tcpIpConfig_2 = config.getNetworkConfig().getJoin().getTcpIpConfig();
        _tcpIpConfig_2.setEnabled(false);
      } else {
        MulticastConfig _multicastConfig_2 = config.getNetworkConfig().getJoin().getMulticastConfig();
        _multicastConfig_2.setEnabled(false);
        TcpIpConfig _tcpIpConfig_3 = config.getNetworkConfig().getJoin().getTcpIpConfig();
        _tcpIpConfig_3.setEnabled(true);
        String[] iplist = netConfig.getClusterMemberIPsArray();
        if ((iplist != null)) {
          final Supplier<String> _function_2 = () -> {
            return MessageFormat.format(Messages.NetworkModule_5, Arrays.toString(iplist));
          };
          logger.getKernelLogger().log(Level.FINE, _function_2);
        }
        for (final String memberIp : iplist) {
          config.getNetworkConfig().getJoin().getTcpIpConfig().addMember(memberIp);
        }
      }
      NetworkConfig _networkConfig = config.getNetworkConfig();
      _networkConfig.setPortAutoIncrement(netConfig.isPortAutoIncrement());
      final int minNumber = netConfig.getMinClusterSize();
      config.setProperty("hazelcast.initial.min.cluster.size", Integer.valueOf(minNumber).toString());
      int otherNumber = (minNumber - 1);
      String startLogMsg = null;
      if ((otherNumber > 1)) {
        startLogMsg = MessageFormat.format(Messages.NetworkModule_2, Integer.valueOf(otherNumber).toString());
      } else {
        if ((otherNumber > 0)) {
          startLogMsg = MessageFormat.format(Messages.NetworkModule_1, Integer.valueOf(otherNumber).toString());
        } else {
          startLogMsg = Messages.NetworkModule_0;
        }
      }
      logger.getKernelLogger().info(startLogMsg);
    }
    final HazelcastInstance hzlInstance = Hazelcast.newHazelcastInstance(config);
    HazelcastLogListener _hazelcastLogListener = new HazelcastLogListener(logger);
    hzlInstance.getLoggingService().addLogListener(logConfig.getLevel().toJul(), _hazelcastLogListener);
    return hzlInstance;
  }

  @SyntheticMember
  public NetworkModule() {
    super();
  }
}
