/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.sre.janus.network.services;

import com.hazelcast.collection.ISet;
import com.hazelcast.collection.ItemEvent;
import com.hazelcast.collection.ItemListener;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.topic.ITopic;
import com.hazelcast.topic.Message;
import com.hazelcast.topic.MessageListener;
import io.sarl.api.core.spaces.AbstractEventSpace;
import io.sarl.api.core.spaces.EventTransportService;
import io.sarl.api.naming.name.SpaceName;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.Event;
import io.sarl.lang.core.EventSpace;
import io.sarl.lang.core.Scope;
import io.sarl.lang.core.Space;
import io.sarl.lang.core.SpaceID;
import io.sarl.lang.core.annotation.Injectable;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import io.sarl.lang.core.util.ConcurrentCollection;
import io.sarl.sre.janus.services.context.Context;
import io.sarl.sre.janus.services.context.ContextService;
import io.sarl.sre.janus.services.logging.LoggingService;
import jakarta.inject.Inject;
import jakarta.inject.Singleton;
import java.io.Serializable;
import java.text.MessageFormat;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Implementation of a MTS using Hazeclast topics to distribute messages over a network of SARL's SREs.
 * For EventSpace, the Hazelcast topics' messages contains events
 * 
 * <p>Topics' naming conventions (based on io.sarl.api.naming): <ul>
 * <li>Context: io.sarl.topics.context.contextId</li>
 * <li>Space: io.sarl.topics.space.contextId.spaceId</li>
 * <li>Agent: io.sarl.topics.agent.contextId.spaceId.agentId</li>
 * <li>Skill: io.sarl.topics.skill.contextId.spaceId.agentId.capacityType</li>
 * <li>Behavior: io.sarl.topics.behavior.contextId.spaceId.agentId.behaviorType.behaviorIndex</li>
 * <li>Skill: io.sarl.topics.service.serviceType</li>
 * </ul>
 * 
 * @author <a href="http://www.ciad-lab.fr/nicolas_gaud">Nicolas Gaud</a>
 * @version janus.network 3.0.15.1 20250911-224826
 * @mavengroupid io.sarl.sre.janus
 * @mavenartifactid janus.network
 * @since 0.12
 */
@Singleton
@SarlSpecification("0.15")
@SarlElementType(10)
@Injectable
@XbaseGenerated
@SuppressWarnings("all")
public class HazelcastEventTransportService implements EventTransportService {
  /**
   * Envelope for Hazelcast messages.
   * 
   * @author <a href="http://www.ciad-lab.fr/nicolas_gaud">Nicolas Gaud</a>
   * @version janus.network 3.0.15.1 20250911-224826
   * @mavengroupid io.sarl.sre.janus
   * @mavenartifactid janus.network
   * @since 0.12
   */
  @SarlSpecification("0.15")
  @SarlElementType(10)
  @XbaseGenerated
  public static class TopicMessage implements Serializable {
    @Accessors
    private final Event transferredEvent;

    @Accessors
    private final Scope<? super Address> trasnferredScope;

    public TopicMessage(final Event e, final Scope<? super Address> s) {
      this.transferredEvent = e;
      this.trasnferredScope = s;
    }

    @Override
    @Pure
    @SyntheticMember
    public boolean equals(final Object obj) {
      return super.equals(obj);
    }

    @Override
    @Pure
    @SyntheticMember
    public int hashCode() {
      int result = super.hashCode();
      return result;
    }

    @SyntheticMember
    private static final long serialVersionUID = 510724359L;

    @Pure
    public Event getTransferredEvent() {
      return this.transferredEvent;
    }

    @Pure
    public Scope<? super Address> getTrasnferredScope() {
      return this.trasnferredScope;
    }
  }

  /**
   * Listener on Hazelcast events.
   * 
   * @author <a href="http://www.ciad-lab.fr/nicolas_gaud">Nicolas Gaud</a>
   * @version janus.network 3.0.15.1 20250911-224826
   * @mavengroupid io.sarl.sre.janus
   * @mavenartifactid janus.network
   * @since 0.12
   */
  @SarlSpecification("0.15")
  @SarlElementType(10)
  @XbaseGenerated
  protected static class TopicNameListener implements ItemListener<String> {
    /**
     * SRE Kernel logger
     */
    private Logger kernelLogger;

    /**
     * Local Hazelcast instance
     */
    private HazelcastInstance hazelcastInstance;

    /**
     * Map associating to each locally defined topic its corresponding listener ID
     */
    private ConcurrentHashMap<String, UUID> topicListenerIDs;

    /**
     * ContextService used to check if the enclosing context exists when creating a new local space according to netwrok space creation event
     */
    private ContextService contextService;

    public TopicNameListener(final Logger logger, final HazelcastInstance hzl, final ConcurrentHashMap<String, UUID> itopicListenerIDs, final ContextService icontextService) {
      this.kernelLogger = logger;
      this.hazelcastInstance = hzl;
      this.topicListenerIDs = itopicListenerIDs;
      this.contextService = icontextService;
    }

    public void itemAdded(final ItemEvent<String> newTopicName) {
      this.kernelLogger.log(Level.INFO, MessageFormat.format(Messages.TopicNameListener_0, newTopicName.getItem()));
      boolean _containsKey = this.topicListenerIDs.containsKey(newTopicName.getItem());
      if ((!_containsKey)) {
        this.kernelLogger.log(Level.INFO, MessageFormat.format(Messages.TopicNameListener_1, newTopicName.getItem()));
        ITopic<HazelcastEventTransportService.TopicMessage> topic = this.hazelcastInstance.<HazelcastEventTransportService.TopicMessage>getReliableTopic(newTopicName.getItem());
        String _item = newTopicName.getItem();
        HazelcastEventTransportService.TopicMessageListener _topicMessageListener = new HazelcastEventTransportService.TopicMessageListener(this.kernelLogger, _item, this.contextService);
        UUID listenerID = topic.addMessageListener(_topicMessageListener);
        this.topicListenerIDs.put(newTopicName.getItem(), listenerID);
      }
    }

    public void itemRemoved(final ItemEvent<String> removedTopicName) {
      this.kernelLogger.log(Level.INFO, MessageFormat.format(Messages.TopicNameListener_2, removedTopicName.getItem()));
      ITopic<Event> topic = this.hazelcastInstance.<Event>getReliableTopic(removedTopicName.getItem());
      UUID existingListenerID = this.topicListenerIDs.get(removedTopicName.getItem());
      topic.removeMessageListener(existingListenerID);
      this.topicListenerIDs.remove(existingListenerID);
    }

    @Override
    @Pure
    @SyntheticMember
    public boolean equals(final Object obj) {
      return super.equals(obj);
    }

    @Override
    @Pure
    @SyntheticMember
    public int hashCode() {
      int result = super.hashCode();
      return result;
    }
  }

  /**
   * Envelope for Hazelcast topic messages.
   * 
   * @author <a href="http://www.ciad-lab.fr/nicolas_gaud">Nicolas Gaud</a>
   * @version janus.network 3.0.15.1 20250911-224826
   * @mavengroupid io.sarl.sre.janus
   * @mavenartifactid janus.network
   * @since 0.12
   */
  @SarlSpecification("0.15")
  @SarlElementType(10)
  @XbaseGenerated
  protected static class TopicMessageListener implements MessageListener<HazelcastEventTransportService.TopicMessage> {
    /**
     * SRE Kernel logger
     */
    private Logger kernelLogger;

    private String listenTopicName;

    /**
     * ContextService used to check if the enclosing context exists when creating a new local space according to netwrok space creation event
     */
    private ContextService contextService;

    public TopicMessageListener(final Logger logger, final String topicName, final ContextService icontextService) {
      this.kernelLogger = logger;
      this.listenTopicName = topicName;
      this.contextService = icontextService;
    }

    public void onMessage(final Message<HazelcastEventTransportService.TopicMessage> incomingMessage) {
      Event eventToDispatchLocally = incomingMessage.getMessageObject().transferredEvent;
      String contextUIIDString = this.listenTopicName.substring(21, 57);
      String spaceUIIDString = this.listenTopicName.substring(58, 94);
      UUID contextUUID = UUID.fromString(contextUIIDString);
      UUID spaceUIID = UUID.fromString(spaceUIIDString);
      Context context = this.contextService.getContext(contextUUID);
      if ((context != null)) {
        ConcurrentCollection<? extends Space> _spaces = context.getSpaces();
        for (final Space localSpace : _spaces) {
          boolean _equals = localSpace.getSpaceID().getID().equals(spaceUIID);
          if (_equals) {
            this.kernelLogger.log(Level.INFO, MessageFormat.format(Messages.TopicMessageListener_0, spaceUIID, contextUUID));
            ((AbstractEventSpace) localSpace).emit(eventToDispatchLocally.getSource().getID(), eventToDispatchLocally, ((Scope<Address>) incomingMessage.getMessageObject().trasnferredScope));
          }
        }
      } else {
        this.kernelLogger.log(Level.SEVERE, MessageFormat.format(Messages.TopicMessageListener_1, spaceUIID, contextUUID));
      }
    }

    @Override
    @Pure
    @SyntheticMember
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      TopicMessageListener other = (TopicMessageListener) obj;
      if (!Objects.equals(this.listenTopicName, other.listenTopicName))
        return false;
      return super.equals(obj);
    }

    @Override
    @Pure
    @SyntheticMember
    public int hashCode() {
      int result = super.hashCode();
      final int prime = 31;
      result = prime * result + Objects.hashCode(this.listenTopicName);
      return result;
    }
  }

  /**
   * Local Hazelcast instance
   */
  @Accessors(AccessorType.PUBLIC_GETTER)
  private HazelcastInstance hazelcastInstance;

  /**
   * Distributed Set of all existing topics defined with the network of existing SARL SRE instances
   */
  private ISet<String> availableTopicsName;

  /**
   * Map associating to each locally defined topic its corresponding listener ID
   */
  private ConcurrentHashMap<String, UUID> topicListenerIDs;

  /**
   * ContextService used to check if the enclosing context exists when creating a new local space according to netwrok space creation event
   */
  private ContextService contextService;

  /**
   * SRE Kernel logger
   */
  private Logger kernelLogger;

  public static final String HAZELCAST_SARL_TOPICS_NAME_SET = "io.sarl.topics.distributedset";

  public static final String HAZELCAST_SARL_TOPICS_ROOTWILCARD = "io.sarl.topics.";

  @Inject
  public HazelcastEventTransportService(final HazelcastInstance iHazelcastInstance, final ContextService icontextService, final LoggingService logger) {
    this.contextService = icontextService;
    this.hazelcastInstance = iHazelcastInstance;
    this.kernelLogger = logger.getKernelModuleLogger(Messages.HazelcastEventTransportService_1);
    ConcurrentHashMap<String, UUID> _concurrentHashMap = new ConcurrentHashMap<String, UUID>();
    this.topicListenerIDs = _concurrentHashMap;
    this.availableTopicsName = this.hazelcastInstance.<String>getSet(HazelcastEventTransportService.HAZELCAST_SARL_TOPICS_NAME_SET);
    HazelcastEventTransportService.TopicNameListener _topicNameListener = new HazelcastEventTransportService.TopicNameListener(this.kernelLogger, this.hazelcastInstance, this.topicListenerIDs, this.contextService);
    this.availableTopicsName.addItemListener(_topicNameListener, true);
    String defaultSpaceRootContextTopicName = HazelcastEventTransportService.getTopicNameFromSpaceID(icontextService.getRootContext().getDefaultSpace().getSpaceID());
    this.availableTopicsName.add(defaultSpaceRootContextTopicName);
    ITopic<HazelcastEventTransportService.TopicMessage> defaultSpaceRootContextTopic = this.hazelcastInstance.<HazelcastEventTransportService.TopicMessage>getReliableTopic(defaultSpaceRootContextTopicName);
    HazelcastEventTransportService.TopicMessageListener _topicMessageListener = new HazelcastEventTransportService.TopicMessageListener(this.kernelLogger, defaultSpaceRootContextTopicName, this.contextService);
    UUID listenerID = defaultSpaceRootContextTopic.addMessageListener(_topicMessageListener);
    this.topicListenerIDs.put(defaultSpaceRootContextTopicName, listenerID);
  }

  public boolean routeEvent(final Event event, final EventSpace space, final Scope<? super Address> scope) {
    boolean _xblockexpression = false;
    {
      String topicName = HazelcastEventTransportService.getTopicNameFromSpaceID(space.getSpaceID());
      ITopic<HazelcastEventTransportService.TopicMessage> spaceTopic = this.hazelcastInstance.<HazelcastEventTransportService.TopicMessage>getReliableTopic(topicName);
      this.kernelLogger.log(Level.INFO, MessageFormat.format(Messages.HazelcastEventTransportService_0, 
        event.getClass(), space.getSpaceID().getID(), space.getSpaceID().getContextID(), topicName));
      HazelcastEventTransportService.TopicMessage _topicMessage = new HazelcastEventTransportService.TopicMessage(event, scope);
      spaceTopic.publish(_topicMessage);
      _xblockexpression = true;
    }
    return _xblockexpression;
  }

  @Pure
  public static String getTopicNameFromSpaceID(final SpaceID spaceID) {
    UUID _contextID = spaceID.getContextID();
    UUID _iD = spaceID.getID();
    return (((((HazelcastEventTransportService.HAZELCAST_SARL_TOPICS_ROOTWILCARD + SpaceName.SCHEME) + ".") + _contextID) + ".") + _iD);
  }

  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    return super.equals(obj);
  }

  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    return result;
  }

  @Pure
  public HazelcastInstance getHazelcastInstance() {
    return this.hazelcastInstance;
  }
}
