/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2025 SARL.io, the Original Authors and Main Authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.sre.janus.network.services;

import com.hazelcast.collection.ISet;
import com.hazelcast.core.EntryEvent;
import com.hazelcast.core.EntryListener;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.map.IMap;
import com.hazelcast.map.MapEvent;
import io.bootique.di.Injector;
import io.sarl.api.naming.name.SpaceName;
import io.sarl.lang.core.Space;
import io.sarl.lang.core.SpaceID;
import io.sarl.lang.core.SpaceSpecification;
import io.sarl.lang.core.annotation.DefaultValue;
import io.sarl.lang.core.annotation.DefaultValueSource;
import io.sarl.lang.core.annotation.DefaultValueUse;
import io.sarl.lang.core.annotation.Injectable;
import io.sarl.lang.core.annotation.SarlElementType;
import io.sarl.lang.core.annotation.SarlSourceCode;
import io.sarl.lang.core.annotation.SarlSpecification;
import io.sarl.lang.core.annotation.SyntheticMember;
import io.sarl.sre.janus.boot.configs.SreConfig;
import io.sarl.sre.janus.internal.Factories;
import io.sarl.sre.janus.services.context.Context;
import io.sarl.sre.janus.services.context.ContextService;
import io.sarl.sre.janus.services.context.LocalSpaceRepository;
import io.sarl.sre.janus.services.context.SpaceRepository;
import io.sarl.sre.janus.services.executor.ExecutorService;
import io.sarl.sre.janus.services.logging.LoggingService;
import io.sarl.sre.janus.spaces.SreSpaceSpecificationFactory;
import jakarta.inject.Inject;
import jakarta.inject.Provider;
import java.text.MessageFormat;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Implementation of a space repository that is connected to remote SRE with Hazelcast framework.
 * 
 * @author <a href="http://www.ciad-lab.fr/nicolas_gaud">Nicolas Gaud</a>
 * @author <a href="http://www.ciad-lab.fr/stephane_galland">St&eacute;phane Galland</a>
 * @version janus.network 3.0.15.1 20250911-224826
 * @mavengroupid io.sarl.sre.janus
 * @mavenartifactid janus.network
 * @since 0.12
 */
@SarlSpecification("0.15")
@SarlElementType(10)
@Injectable
@XbaseGenerated
@SuppressWarnings("all")
public class HazelcastSpaceRepository extends LocalSpaceRepository {
  /**
   * Local Hazelcast instance
   */
  @Accessors(AccessorType.PUBLIC_GETTER)
  private HazelcastInstance hazelcastInstance;

  /**
   * The set of the id of all spaces stored in this repository This set must be
   * distributed and synchronized all over the network.
   */
  private IMap<SpaceID, Object[]> spaceIDs;

  /**
   * ContextService used to check if the enclosing context exists when creating a new local space according to netwrok space creation event
   */
  private ContextService contextService;

  /**
   * ID of the listener defined on spaceIDs map
   */
  private UUID spaceIDsListernerID;

  /**
   * Set of existing distributed topics (at least one per space)
   */
  private ISet<String> availableTopicsName;

  public static final String HAZELCAST_SPACE_ID_MAP_NAME = "io.sarl.network.distributedSpaceIDMapName";

  @DefaultValueSource
  @Inject
  public HazelcastSpaceRepository(final Injector injector, final ExecutorService executor, final LoggingService loggingService, final SreConfig config, final SreSpaceSpecificationFactory specificationFactory, final ContextService icontextService, final Provider<Factories> factories, final HazelcastInstance iHazelcastInstance, @DefaultValue("io.sarl.sre.janus.network.services.HazelcastSpaceRepository#NEW_0") final Map<UUID, SpaceRepository.SpaceDescription> internalStructure0, @DefaultValue("io.sarl.sre.janus.network.services.HazelcastSpaceRepository#NEW_1") final IMap<SpaceID, Object[]> internalStructure1) {
    super(injector, executor, loggingService, config, specificationFactory, internalStructure0, factories);
    this.contextService = icontextService;
    this.hazelcastInstance = iHazelcastInstance;
    this.logger.log(Level.INFO, Messages.HazelcastSpaceRepository_0);
    this.availableTopicsName = this.hazelcastInstance.<String>getSet(
      HazelcastEventTransportService.HAZELCAST_SARL_TOPICS_NAME_SET);
    IMap<SpaceID, Object[]> _elvis = null;
    if (internalStructure1 != null) {
      _elvis = internalStructure1;
    } else {
      IMap<SpaceID, Object[]> _map = this.hazelcastInstance.<SpaceID, Object[]>getMap(HazelcastSpaceRepository.HAZELCAST_SPACE_ID_MAP_NAME);
      _elvis = _map;
    }
    this.spaceIDs = _elvis;
    EntryListener<SpaceID, Object[]> spaceIDsListerner = new EntryListener<SpaceID, Object[]>() {
      @Override
      public void entryAdded(final EntryEvent<SpaceID, Object[]> event) {
        class $AssertEvaluator$ {
          final boolean $$result;
          $AssertEvaluator$() {
            this.$$result = HazelcastSpaceRepository.this.spaceIDs.containsKey(event.getKey());
          }
        }
        assert new $AssertEvaluator$().$$result;
        HazelcastSpaceRepository.this.<Space>ensureLocalSpaceDefinition(event.getKey(), event.getValue());
      }

      @Override
      public void entryEvicted(final EntryEvent<SpaceID, Object[]> event) {
        class $AssertEvaluator$ {
          final boolean $$result;
          $AssertEvaluator$() {
            boolean _containsKey = HazelcastSpaceRepository.this.spaceIDs.containsKey(event.getKey());
            this.$$result = (!_containsKey);
          }
        }
        assert new $AssertEvaluator$().$$result;
        HazelcastSpaceRepository.this.removeLocalSpaceDefinition(event.getKey(), false);
      }

      @Override
      public void entryRemoved(final EntryEvent<SpaceID, Object[]> event) {
        class $AssertEvaluator$ {
          final boolean $$result;
          $AssertEvaluator$() {
            boolean _containsKey = HazelcastSpaceRepository.this.spaceIDs.containsKey(event.getKey());
            this.$$result = (!_containsKey);
          }
        }
        assert new $AssertEvaluator$().$$result;
        HazelcastSpaceRepository.this.removeLocalSpaceDefinition(event.getKey(), false);
      }

      @Override
      public void entryUpdated(final EntryEvent<SpaceID, Object[]> event) {
      }

      @Override
      public void mapCleared(final MapEvent event) {
        HazelcastSpaceRepository.this.removeLocalSpaceDefinitions();
      }

      @Override
      public void mapEvicted(final MapEvent event) {
      }

      @Override
      public void entryExpired(final EntryEvent<SpaceID, Object[]> event) {
      }
    };
    this.spaceIDsListernerID = this.spaceIDs.addEntryListener(spaceIDsListerner, true);
    this.logger.log(Level.INFO, Messages.HazelcastSpaceRepository_1);
  }

  @Override
  @Pure
  protected Logger createLogger(final LoggingService loggingService) {
    return loggingService.getKernelModuleLogger(Messages.HazelcastSpaceRepository_5);
  }

  @Override
  protected <S extends Space> S createSpaceFirstInstance(final Class<? extends SpaceSpecification<S>> spec, final SpaceID spaceID, final Object[] creationParams) {
    Context localContext = this.contextService.getContext(spaceID.getContextID());
    if ((localContext == null)) {
      this.logger.log(Level.SEVERE, Messages.HazelcastSpaceRepository_2);
    }
    this.spaceIDs.putIfAbsent(spaceID, creationParams);
    String topicName = HazelcastEventTransportService.getTopicNameFromSpaceID(spaceID);
    this.logger.log(Level.INFO, MessageFormat.format(Messages.HazelcastSpaceRepository_3, spaceID, topicName));
    this.availableTopicsName.add(topicName);
    return super.<S>createSpaceFirstInstance(spec, spaceID, creationParams);
  }

  /**
   * Add the existing, but not yet known, spaces into this repository.
   * 
   * @param id identifier of the space
   * @param initializationParameters parameters for initialization.
   */
  protected <S extends Space> S ensureLocalSpaceDefinition(final SpaceID id, final Object[] initializationParameters) {
    S _xifexpression = null;
    boolean _hasSpace = this.hasSpace(id.getID());
    if ((!_hasSpace)) {
      S _xblockexpression = null;
      {
        Class<? extends SpaceSpecification<?>> _spaceSpecification = id.getSpaceSpecification();
        Class<? extends SpaceSpecification<S>> spec = ((Class<? extends SpaceSpecification<S>>) _spaceSpecification);
        _xblockexpression = super.<S>createSpace(id, spec, initializationParameters);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }

  protected void removeLocalSpaceDefinition(final SpaceID id, final boolean isLocalDestruction) {
    UUID _contextID = id.getContextID();
    UUID _iD = id.getID();
    String topicName = (((((HazelcastEventTransportService.HAZELCAST_SARL_TOPICS_ROOTWILCARD + SpaceName.SCHEME) + 
      ".") + _contextID) + ".") + _iD);
    this.logger.log(Level.INFO, MessageFormat.format(Messages.HazelcastSpaceRepository_4, topicName));
    this.availableTopicsName.remove(topicName);
    super.removeSpaceIfEmpty(id.getID());
  }

  /**
   * Remove all the local/remote spaces.
   */
  protected void removeLocalSpaceDefinitions() {
    this.removeEmptySpaces();
  }

  @Override
  protected void destroyThreadUnsafe() {
    super.destroyThreadUnsafe();
    this.spaceIDs.clear();
    this.spaceIDs.removeEntryListener(this.spaceIDsListernerID);
  }

  /**
   * Default value for the parameter internalStructure0
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static Map $DEFAULT_VALUE$NEW_0() {
    return null;
  }

  /**
   * Default value for the parameter internalStructure1
   */
  @Pure
  @SyntheticMember
  @SarlSourceCode("null")
  private static IMap $DEFAULT_VALUE$NEW_1() {
    return null;
  }

  @DefaultValueUse("io.bootique.di.Injector,io.sarl.sre.janus.services.executor.ExecutorService,io.sarl.sre.janus.services.logging.LoggingService,io.sarl.sre.janus.boot.configs.SreConfig,io.sarl.sre.janus.spaces.SreSpaceSpecificationFactory,io.sarl.sre.janus.services.context.ContextService,jakarta.inject.Provider,com.hazelcast.core.HazelcastInstance,java.util.Map,com.hazelcast.map.IMap")
  @SyntheticMember
  public HazelcastSpaceRepository(final Injector injector, final ExecutorService executor, final LoggingService loggingService, final SreConfig config, final SreSpaceSpecificationFactory specificationFactory, final ContextService icontextService, final Provider<Factories> factories, final HazelcastInstance iHazelcastInstance) {
    this(injector, executor, loggingService, config, specificationFactory, icontextService, factories, iHazelcastInstance, $DEFAULT_VALUE$NEW_0(), $DEFAULT_VALUE$NEW_1());
  }

  @DefaultValueUse("io.bootique.di.Injector,io.sarl.sre.janus.services.executor.ExecutorService,io.sarl.sre.janus.services.logging.LoggingService,io.sarl.sre.janus.boot.configs.SreConfig,io.sarl.sre.janus.spaces.SreSpaceSpecificationFactory,io.sarl.sre.janus.services.context.ContextService,jakarta.inject.Provider,com.hazelcast.core.HazelcastInstance,java.util.Map,com.hazelcast.map.IMap")
  @SyntheticMember
  public HazelcastSpaceRepository(final Injector injector, final ExecutorService executor, final LoggingService loggingService, final SreConfig config, final SreSpaceSpecificationFactory specificationFactory, final ContextService icontextService, final Provider<Factories> factories, final HazelcastInstance iHazelcastInstance, final IMap<SpaceID, Object[]> internalStructure1) {
    this(injector, executor, loggingService, config, specificationFactory, icontextService, factories, iHazelcastInstance, $DEFAULT_VALUE$NEW_0(), internalStructure1);
  }

  @DefaultValueUse("io.bootique.di.Injector,io.sarl.sre.janus.services.executor.ExecutorService,io.sarl.sre.janus.services.logging.LoggingService,io.sarl.sre.janus.boot.configs.SreConfig,io.sarl.sre.janus.spaces.SreSpaceSpecificationFactory,io.sarl.sre.janus.services.context.ContextService,jakarta.inject.Provider,com.hazelcast.core.HazelcastInstance,java.util.Map,com.hazelcast.map.IMap")
  @SyntheticMember
  public HazelcastSpaceRepository(final Injector injector, final ExecutorService executor, final LoggingService loggingService, final SreConfig config, final SreSpaceSpecificationFactory specificationFactory, final ContextService icontextService, final Provider<Factories> factories, final HazelcastInstance iHazelcastInstance, final Map<UUID, SpaceRepository.SpaceDescription> internalStructure0) {
    this(injector, executor, loggingService, config, specificationFactory, icontextService, factories, iHazelcastInstance, internalStructure0, $DEFAULT_VALUE$NEW_1());
  }

  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    HazelcastSpaceRepository other = (HazelcastSpaceRepository) obj;
    if (!Objects.equals(this.spaceIDsListernerID, other.spaceIDsListernerID))
      return false;
    return super.equals(obj);
  }

  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.spaceIDsListernerID);
    return result;
  }

  @Pure
  public HazelcastInstance getHazelcastInstance() {
    return this.hazelcastInstance;
  }
}
